<?php
/**
 * Form to reset password
 */

if(!defined('ABSPATH')) {
	exit;
}

// Enqueue styles & scripts
function ejabat_enqueue_reset_password_scripts() {
	global $post;
	if(is_a($post, 'WP_Post') && has_shortcode($post->post_content, 'ejabat_reset_password')) {
		wp_enqueue_style('ejabat', EJABAT_DIR_URL.'css/style.min.css', array(), EJABAT_VERSION, 'all');
		wp_enqueue_script('ejabat-form', EJABAT_DIR_URL.'js/js.ejabat.form.min.js', array(), EJABAT_VERSION, true);
		if(get_option('ejabat_show_hints', true)) {
			$hints = apply_filters('ejabat_hints_args', array(
				'password' => get_option('ejabat_password_hint', __('At least a good password is required.', 'ejabberd-account-tools'))
			));
		}
		wp_localize_script('ejabat-form', 'ejabat', array(
			'nonce' => ejabat_create_nonce(),
			'rest_api' => esc_url_raw(rest_url().'ejabberd-account-tools/v1/'),
			'password_strength' => get_option('ejabat_password_strength', 3),
			'password_too_weak' => __('The password is too weak.', 'ejabberd-account-tools'),
			'password_very_weak' => __('The password is very weak.', 'ejabberd-account-tools'),
			'password_weak' => __('The password is weak.', 'ejabberd-account-tools'),
			'password_good' => __('The password is good.', 'ejabberd-account-tools'),
			'password_strong' => __('The password is strong.', 'ejabberd-account-tools'),
			'empty_field' => __('Please complete the required field.', 'ejabberd-account-tools'),
			'empty_fields' => __('Verification errors occurred. Please check all fields and submit the form again.', 'ejabberd-account-tools'),
			'password_hint' => $hints['password'],
			'error' => __('An unexpected error occurred. Please try again.', 'ejabberd-account-tools'),
			'form_error' => '<p class="ejabat"><span class="ejabat-info ejabat-error">'.__('An unexpected error occurred. Please try again.', 'ejabberd-account-tools').'</span></p>'
		));
		wp_enqueue_script('zxcvbn-async');
	}
}
add_action('wp_enqueue_scripts', 'ejabat_enqueue_reset_password_scripts');

// Reset password shortcode
function ejabat_reset_password_shortcode() {
	return '<p data-action="reset-password-form" class="ejabat">'.(get_option('ejabat_loader', true) && true ? '<span class="ejabat-loader" title="'.__('Loading', 'ejabberd-account-tools').'..."></span>' : '').'</p>';
}
add_shortcode('ejabat_reset_password', 'ejabat_reset_password_shortcode');

// Route reset password form
function ejabat_route_reset_password_form() {
	register_rest_route('ejabberd-account-tools/v1', '/reset-password-form', array(
		'methods' => 'POST',
		'callback' => 'ejabat_reset_password_form',
		'permission_callback' => '__return_true',
		'args' => array(
			'code' => array(
				'type' => 'string',
				'default' => 'undefined',
				'sanitize_callback' => function($param, $request, $key) {
					return stripslashes_deep(sanitize_text_field($param));
				}
			)
		)
	));
}
add_action('rest_api_init', 'ejabat_route_reset_password_form');

// Reset password form
function ejabat_reset_password_form($request) {
	// Form is disabled
	if(get_option('ejabat_disable_reset_pass', false) && !is_user_logged_in()) {
		$html = '<p class="ejabat"><span class="ejabat-info ejabat-error">'.__('The form to reset password is temporarily disabled. Please try again later.', 'ejabberd-account-tools').'</span></p>';
	}
	else {
		// Link to reset password
		if($request['code'] != 'undefined') {
			// Code valid
			if(true == ($transient = get_transient('ejabat_pass_'.$request['code']))) {
				// Create form
				$html = '<form data-action="change-password" class="ejabat" method="post" onsubmit="return false" autocomplete="off">
					<p id="login">
						<input type="text" name="login" value="'.$transient['user'].'@'.$transient['host'].'" disabled>
						<span class="ejabat-tip"></span>
					</p>
					<p id="password" class="ejabat-strength ejabat-validate ejabat-hint">
						<input type="password" name="password" placeholder="'.__('Password', 'ejabberd-account-tools').'">
						<span class="ejabat-tip"></span>
					</p>
					<p>
						<input type="submit" value="'.__('Set a new password', 'ejabberd-account-tools').'">
						<input type="hidden" name="code" value="'.$request['code'].'">
						<span class="ejabat-spinner" style="visibility: hidden;"></span>
					</p>
					<div id="response"></div>
				</form>';
			}
			// Code expired or not valid
			else {
				// Delete transient
				delete_transient('ejabat_pass_'.$code);
				// Response with error
				$message = '<p id="message" class="ejabat-info ejabat-blocked">'.__('The link to reset your password has expired or is invalid. Please complete the form and resubmit it.', 'ejabberd-account-tools').'</p>';
			}
		}
		// Create form
		$html = isset($html) ? $html : '<form data-action="reset-password" class="ejabat" method="post" onsubmit="return false" autocomplete="off">
			'.(isset($message) ? $message : '').'
			<p id="login" class="ejabat-validate">
				<input type="text" name="login" placeholder="'.__('Full username', 'ejabberd-account-tools').'">
				<span class="ejabat-tip"></span>
			</p>
			'.ejabat_captcha_field().'
			<p>
				<input type="submit" value="'.__('Reset password', 'ejabberd-account-tools').'">
				<span class="ejabat-spinner" style="visibility: hidden;"></span>
			</p>
			<div id="response"></div>
		</form>';
	}
	return rest_ensure_response(array('data' => str_replace(array(PHP_EOL, "\t"), '', $html), 'nonce' => wp_create_nonce('wp_rest')));
}

// Route reset password
function ejabat_route_reset_password() {
	register_rest_route('ejabberd-account-tools/v1', '/reset-password', array(
		'methods' => 'POST',
		'callback' => 'ejabat_reset_password',
		'permission_callback' => '__return_true',
		'args' => array(
			'login' => array(
				'type' => 'string',
				'required' => true,
				'sanitize_callback' => function($param, $request, $key) {
					return stripslashes_deep(sanitize_text_field($param));
				}
			)
		)
	));
}
add_action('rest_api_init', 'ejabat_route_reset_password');

// Reset password callback
function ejabat_reset_password($request) {
	// Verify nonce
	if(!wp_verify_nonce($request->get_header('x-wp-nonce'), 'wp_rest')) {
		$status = 'blocked';
		$message = __('Verification error. Please try again.', 'ejabberd-account-tools');
	}
	else {
		// Verify captcha
		if(true !== ($captcha_verify = ejabat_captcha_verify($request))) {
			$status = 'blocked';
			if(is_array($captcha_verify)) $fields = $captcha_verify;
			$message = __('Captcha validation error. Please try again.', 'ejabberd-account-tools');
		}
		else {
			// Check username
			if(!filter_var($request['login'], FILTER_VALIDATE_EMAIL)) {
				$status = 'blocked';
				$fields = array('login');
				$message = __('Invalid username. Please correct it and try again.', 'ejabberd-account-tools');
			}
			else {
				// Check host
				list($user, $host) = explode('@', $request['login']);
				if(false === ($transient = get_transient('ejabat_registered_vhosts'))) {
					$response = ejabat_get_xmpp_data('registered_vhosts');
					set_transient('ejabat_registered_vhosts', $response['body']);
					$transient = $response['body'];
				}
				if(false === array_search($host, json_decode($transient))) {
					$status = 'blocked';
					$fields = array('login');
					$message = __('Invalid username. Please correct it and try again.', 'ejabberd-account-tools');
				}
				else {
					// Check login
					$response = ejabat_get_xmpp_data('check_account', array('user' => $user, 'host' => $host));
					// Server unavailable
					if(is_null($response)) {
						$status = 'error';
						$message = __('The server is temporarily unavailable. Please try again later.', 'ejabberd-account-tools');
					}
					// User not found
					else if($response['code'] == 1) {
						$status = 'blocked';
						$fields = array('login');
						$message = __('Invalid username. Please correct it and try again.', 'ejabberd-account-tools');
					}
					// User found
					else if($response['code'] == 0) {
						// Check if account is banned
						$response = ejabat_get_xmpp_data('get_ban_details', array('user' => $user, 'host' => $host));
						if(!isset(json_decode($response['body'])->reason) || (isset(json_decode($response['body'])->reason) ? json_decode($response['body'])->reason : '') == 'Activation required') {
							// Get private email address
							$response = ejabat_get_xmpp_data('private_get', array('user' => $user, 'host' => $host, 'element' => 'private', 'ns' => 'email'));
							// Server unavailable
							if(is_null($response)) {
								$status = 'error';
								$message = __('The server is temporarily unavailable. Please try again later.', 'ejabberd-account-tools');
							}
							// Check response
							else if($response['code'] == 0) {
								// Private email set
								if(true == ($email = json_decode(strip_tags($response['body'])))) {
									// Check verification limit transient
									if(true == ($transient = get_transient('ejabat_pass_'.$user.'@'.$host))) {
										$count = $transient['count'];
									}
									// Verification limit is not exceeded
									if((isset($count) ? $count : 0) < get_option('ejabat_reset_pass_limit_count', 4)) {
										// Get current timestamp
										$now = time();
										// Set verification limit transient
										$data = array('timestamp' => $now, 'ip' => $_SERVER['REMOTE_ADDR'], 'count' => (isset($count) ? $count : 0) + 1);
										set_transient('ejabat_pass_'.$user.'@'.$host, $data, get_option('ejabat_reset_pass_limit_timeout', 43200));
										// Set code transient
										$code = bin2hex(openssl_random_pseudo_bytes(16));
										$data = array('timestamp' => $now, 'ip' => $_SERVER['REMOTE_ADDR'], 'user' => $user, 'host' => $host, 'email' => $email);
										set_transient('ejabat_pass_'.$code, $data, get_option('ejabat_reset_pass_timeout', 900));
										// Email data
										$subject = sprintf(__('Password reset for your account on %s', 'ejabberd-account-tools'), $host);
										$body = sprintf(__('Hey %s!<br><br>Someone requested to change the password for your XMPP account %s. To complete the change, please click the following link:<br><br>%s<br><br>If you haven\'t made this change, simply ignore this email.<br><br>Best regards,<br>%s', 'ejabberd-account-tools'), ejabbat_get_vcard_name($user, $host), $user.'@'.$host, '<a href="'.explode('?', $_SERVER['HTTP_REFERER'])[0].'?code='.$code.'">'.explode('?', $_SERVER['HTTP_REFERER'])[0].'?code='.$code.'</a>', get_bloginfo('name'));
										$headers[] = 'From: '.get_bloginfo('name').' <'.get_option('admin_email').'>';
										$headers[] = 'Content-Type: text/html; charset=UTF-8';
										// Try send email
										if(wp_mail($user.' <'.$email.'>', $subject, $body, $headers)) {
											// Password reset process watcher
											if(get_option('ejabat_watch_reset_pass', false) && get_option('ejabat_watcher')) {
												$watchers = explode(' ', get_option('ejabat_watcher'));
												foreach($watchers as $watcher) {
													ejabat_get_xmpp_data('send_message', array('type' => 'chat', 'from' => $host, 'to' => $watcher, 'subject' => '', 'body' => sprintf('[%s] User %s has requested from IP address %s to reset the password, an email was sent to %s', wp_date('Y-m-d H:i:s', $now), $user.'@'.$host, $_SERVER['REMOTE_ADDR'], $email)));
												}
											}
											// Success message
											$status = 'success';
											$message = sprintf(__('An email has been sent to you at %s. It contains a link to a page where you can reset your password.', 'ejabberd-account-tools'), mask_email($email));
										}
										// Problem with sending email
										else {
											// Delete transient
											delete_transient('ejabat_pass_'.$code);
											// Error message
											$status = 'error';
											$message = __('Failed to send the email. Please try again.', 'ejabberd-account-tools');
										}
									}
									// Verification limit exceeded
									else {
										$status = 'blocked';
										$message = __('Verification limit exceeded. Please try again later.', 'ejabberd-account-tools');
									}
								}
								// Private email not set
								else {
									$status = 'error';
									$message = __('A private email address hasn\'t been set. To reset your password, please contact the administrator.', 'ejabberd-account-tools');
								}
							}
						}
					}
				}
			}
		}
	}
	// Return response
	if(get_option('ejabat_debug', false) == false) return rest_ensure_response(array('status' => isset($status) ? $status : 'error', 'message' => isset($message) ? $message : __('An unexpected error occurred. Please try again.', 'ejabberd-account-tools'), 'fields' => isset($fields) ? $fields : null));
	else return rest_ensure_response(array('status' => isset($status) ? $status : 'error', 'message' => isset($message) ? $message : __('An unexpected error occurred. Please try again.', 'ejabberd-account-tools'), 'fields' => isset($fields) ? $fields : null, 'debug_message' => isset($response['body']) ? $response['body'] : null, 'debug_code' => isset($response['code']) ? $response['code'] : null, 'debug_command' => isset($response['command']) ? $response['command'] : null, 'debug_arguments' => isset($response['arguments']) ? $response['arguments'] : null));
}

// Route change password
function ejabat_route_change_password() {
	register_rest_route('ejabberd-account-tools/v1', '/change-password', array(
		'methods' => 'POST',
		'callback' => 'ejabat_change_password',
		'permission_callback' => '__return_true',
		'args' => array(
			'code' => array(
				'type' => 'string',
				'required' => true,
				'sanitize_callback' => function($param, $request, $key) {
					return stripslashes_deep(sanitize_text_field($param));
				}
			),
			'password' => array(
				'type' => 'string',
				'required' => true,
				'sanitize_callback' => function($param, $request, $key) {
					return stripslashes_deep(sanitize_text_field($param));
				}
			)
		)
	));
}
add_action('rest_api_init', 'ejabat_route_change_password');

// Change password callback
function ejabat_change_password($request) {
	// Verify nonce
	if(!wp_verify_nonce($request->get_header('x-wp-nonce'), 'wp_rest')) {
		$status = 'blocked';
		$message = __('Verification error. Please try again.', 'ejabberd-account-tools');
	}
	else {
		// Code valid
		if(true == ($transient = get_transient('ejabat_pass_'.$request['code']))) {
			// Remove IP address from the fail2ban table
			ejabat_get_xmpp_data('unban_ip', array('address' => $_SERVER['REMOTE_ADDR']));
			// Remove required activation
			$response = ejabat_get_xmpp_data('get_ban_details', array('user' => $transient['user'], 'host' => $transient['host']));
			if((isset(json_decode($response['body'])->reason) ? json_decode($response['body'])->reason : '') == 'Activation required') {
				ejabat_get_xmpp_data('unban_account', array('user' => $transient['user'], 'host' => $transient['host']));
			}
			// Try set new password
			$response = ejabat_get_xmpp_data('change_password', array('user' => $transient['user'], 'host' => $transient['host'], 'newpass' => $request['password']));
			// Server unavailable
			if(is_null($response)) {
				$status = 'error';
				$message = __('The server is temporarily unavailable. Please try again later.', 'ejabberd-account-tools');
			}
			// Password changed
			else if($response['code'] == 0) {
				// Disconnect user's active sessions
				ejabat_get_xmpp_data('kick_user', array('user' => $transient['user'], 'host' => $transient['host']));
				// Password reset process watcher
				if(get_option('ejabat_watch_reset_pass', false) && get_option('ejabat_watcher')) {
					$now = time();
					$watchers = explode(' ', get_option('ejabat_watcher'));
					foreach($watchers as $watcher) {
						ejabat_get_xmpp_data('send_message', array('type' => 'chat', 'from' => $transient['host'], 'to' => $watcher, 'subject' => '', 'body' => sprintf('[%s] User %s has changed the password from IP address %s', wp_date('Y-m-d H:i:s', $now), $transient['user'].'@'.$transient['host'], $_SERVER['REMOTE_ADDR'])));
					}
				}
				// Delete all transients
				delete_transient('ejabat_pass_'.$transient['user'].'@'.$transient['host']);
				delete_transient('ejabat_pass_'.$request['code']);
				// Success message
				$status = 'success';
				$message = __('Your account password has been successfully changed.', 'ejabberd-account-tools');
			}
		}
		// Code expired or not valid
		else {
			// Delete transient
			delete_transient('ejabat_pass_'.$request['code']);
			// Error message
			$status = 'blocked';
			$message = __('The link to reset your password has expired or is invalid.', 'ejabberd-account-tools');
		}
	}
	// Return response
	if(get_option('ejabat_debug', false) == false) return rest_ensure_response(array('status' => isset($status) ? $status : 'error', 'message' => isset($message) ? $message : __('An unexpected error occurred. Please try again.', 'ejabberd-account-tools'), 'fields' => isset($fields) ? $fields : null));
	else return rest_ensure_response(array('status' => isset($status) ? $status : 'error', 'message' => isset($message) ? $message : __('An unexpected error occurred. Please try again.', 'ejabberd-account-tools'), 'fields' => isset($fields) ? $fields : null, 'debug_message' => isset($response['body']) ? $response['body'] : null, 'debug_code' => isset($response['code']) ? $response['code'] : null, 'debug_command' => isset($response['command']) ? $response['command'] : null, 'debug_arguments' => isset($response['arguments']) ? $response['arguments'] : null));
}
