<?php
/**
 * Form to change private email address
 */

if(!defined('ABSPATH')) {
	exit;
}

// Enqueue styles & scripts
function ejabat_enqueue_change_email_scripts() {
	global $post;
	if(is_a($post, 'WP_Post') && has_shortcode($post->post_content, 'ejabat_change_email')) {
		wp_enqueue_style('ejabat', EJABAT_DIR_URL.'css/style.min.css', array(), EJABAT_VERSION, 'all');
		wp_enqueue_script('ejabat-form', EJABAT_DIR_URL.'js/js.ejabat.form.min.js', array(), EJABAT_VERSION, true);
		wp_localize_script('ejabat-form', 'ejabat', array(
			'nonce' => ejabat_create_nonce(),
			'rest_api' => esc_url_raw(rest_url().'ejabberd-account-tools/v1/'),
			'checking_email' => '<span class="ejabat-spinner"></span>'.__('Checking email address...', 'ejabberd-account-tools'),
			'invalid_email' => __('The email address is invalid.', 'ejabberd-account-tools'),
			'did_you_mean' => __('Did you mean', 'ejabberd-account-tools'),
			'empty_field' => __('Please complete the required field.', 'ejabberd-account-tools'),
			'empty_fields' => __('Verification errors occurred. Please check all fields and submit the form again.', 'ejabberd-account-tools'),
			'error' => __('An unexpected error occurred. Please try again.', 'ejabberd-account-tools'),
			'form_error' => '<p class="ejabat"><span class="ejabat-info ejabat-error">'.__('An unexpected error occurred. Please try again.', 'ejabberd-account-tools').'</span></p>'
		));
	}
}
add_action('wp_enqueue_scripts', 'ejabat_enqueue_change_email_scripts');

// Change email shortcode
function ejabat_change_email_shortcode() {
	return '<p data-action="change-email-form" class="ejabat">'.(get_option('ejabat_loader', true) && true ? '<span class="ejabat-loader" title="'.__('Loading', 'ejabberd-account-tools').'..."></span>' : '').'</p>';
}
add_shortcode('ejabat_change_email', 'ejabat_change_email_shortcode');

// Route change email form
function ejabat_route_change_email_form() {
	register_rest_route('ejabberd-account-tools/v1', '/change-email-form', array(
		'methods' => 'POST',
		'callback' => 'ejabat_change_email_form',
		'permission_callback' => '__return_true',
		'args' => array(
			'code' => array(
				'type' => 'string',
				'default' => 'undefined',
				'sanitize_callback' => function($param, $request, $key) {
					return stripslashes_deep(sanitize_text_field($param));
				}
			)
		)
	));
}
add_action('rest_api_init', 'ejabat_route_change_email_form');

// Change email form
function ejabat_change_email_form($request) {
	// Form is disabled
	if(get_option('ejabat_disable_change_email', false) && !is_user_logged_in()) {
		$html = '<p class="ejabat"><span class="ejabat-info ejabat-error">'.__('The form to change private email address is temporarily disabled. Please try again later.', 'ejabberd-account-tools').'</span></p>';
	}
	else {
		// Link to change email
		if($request['code'] != 'undefined') {
			// Code valid
			if(true == ($transient = get_transient('ejabat_email_'.$request['code']))) {
				// Try set private email
				$response = ejabat_get_xmpp_data('private_set', array('user' => $transient['user'], 'host' => $transient['host'], 'element' => '<private xmlns=\'email\'>'.$transient['email'].'</private>'));
				// Server unavailable
				if(is_null($response)) {
					$html = '<p class="ejabat"><span class="ejabat-info ejabat-error">'.__('Changing your private email address failed because the server is temporarily unavailable. Please try again later.', 'ejabberd-account-tools').'</p>';
				}
				// Private email changed
				else if($response['code'] == 0) {
					// Email changes watcher
					if(get_option('ejabat_watch_email_changes', false) && get_option('ejabat_watcher')) {
						$now = time();
						$watchers = explode(' ', get_option('ejabat_watcher'));
						foreach($watchers as $watcher) {
							ejabat_get_xmpp_data('send_message', array('type' => 'chat', 'from' => $transient['host'], 'to' => $watcher, 'subject' => '', 'body' => sprintf('[%s] User %s has changed the email address to %s from IP address %s', wp_date('Y-m-d H:i:s', $now), $transient['user'].'@'.$transient['host'], $transient['email'], $_SERVER['REMOTE_ADDR'])));
						}
					}
					// Delete transient
					delete_transient('ejabat_email_'.$request['code']);
					// Success message
					$html = '<p class="ejabat"><span class="ejabat-info ejabat-success">'.sprintf(__('The private email address for your XMPP account %s has been successfully changed to %s.', 'ejabberd-account-tools'), $transient['user'].'@'.$transient['host'], $transient['email']).'</p>';
				}
				// Unexpected error
				else {
					$html = '<p class="ejabat"><span class="ejabat-info ejabat-error">'.__('An unexpected error occurred while trying to change your private email address. Please try again.', 'ejabberd-account-tools').'</p>';
				}
			}
			// Code expired or not valid
			else {
				delete_transient('ejabat_email_'.$request['code']);
				$message = '<p id="message" class="ejabat-info ejabat-blocked">'.__('The link to change your private email address has expired or is invalid. Please complete the form and resubmit it.', 'ejabberd-account-tools').'</p>';
			}
		}
		// Create form
		$html = isset($html) ? $html : '<form data-action="change-email" class="ejabat" method="post" onsubmit="return false" autocomplete="off">
			'.(isset($message) ? $message : '').'
			<p id="login" class="ejabat-validate">
				<input type="text" name="login" placeholder="'.__('Full username', 'ejabberd-account-tools').'">
				<span class="ejabat-tip"></span>
			</p>
			<p id="password" class="ejabat-validate">
				<input type="password" name="password" placeholder="'.__('Password', 'ejabberd-account-tools').'">
				<span class="ejabat-tip"></span>
			</p>
			<p id="email" class="ejabat-validate">
				<input type="email" name="email" placeholder="'.__('New private email', 'ejabberd-account-tools').'">
				<span class="ejabat-tip"></span>
			</p>
			'.ejabat_captcha_field().'
			<p>
				<input type="submit" value="'.__('Change email', 'ejabberd-account-tools').'">
				<span class="ejabat-spinner" style="visibility: hidden;"></span>
			</p>
			<div id="response"></div>
		</form>';
	}
	return rest_ensure_response(array('data' => str_replace(array(PHP_EOL, "\t"), '', $html), 'nonce' => wp_create_nonce('wp_rest')));
}

// Route change email
function ejabat_route_change_email() {
	register_rest_route('ejabberd-account-tools/v1', '/change-email', array(
		'methods' => 'POST',
		'callback' => 'ejabat_change_email',
		'permission_callback' => '__return_true',
		'args' => array(
			'login' => array(
				'type' => 'string',
				'required' => true,
				'sanitize_callback' => function($param, $request, $key) {
					return stripslashes_deep(sanitize_text_field($param));
				}
			),
			'password' => array(
				'type' => 'string',
				'required' => true,
				'sanitize_callback' => function($param, $request, $key) {
					return stripslashes_deep(sanitize_text_field($param));
				}
			),
			'email' => array(
				'type' => 'string',
				'required' => true,
				'sanitize_callback' => function($param, $request, $key) {
					return stripslashes_deep(sanitize_text_field($param));
				}
			)
		)
	));
}
add_action('rest_api_init', 'ejabat_route_change_email');

// Change email callback
function ejabat_change_email($request) {
	// Verify nonce
	if(!wp_verify_nonce($request->get_header('x-wp-nonce'), 'wp_rest')) {
		$status = 'blocked';
		$message = __('Verification error. Please try again.', 'ejabberd-account-tools');
	}
	else {
		// Verify captcha
		if(true !== ($captcha_verify = ejabat_captcha_verify($request))) {
			$status = 'blocked';
			if(is_array($captcha_verify)) $fields = $captcha_verify;
			$message = __('Captcha validation error. Please try again.', 'ejabberd-account-tools');
		}
		else {
			// Verify email
			if(!filter_var($request['email'], FILTER_VALIDATE_EMAIL) || !ejabat_validate_email_mxrecord($request['email'])) {
				$status = 'blocked';
				$fields = array('email');
				$message = __('The email address is invalid. Please change it and try again.', 'ejabberd-account-tools');
			}
			else if(!ejabat_validate_email_usercheck_com($request['email'])) {
				$status = 'blocked';
				$fields = array('email');
				$message = __('Disposable email addresses are not allowed. Please change the entered email address and try again.', 'ejabberd-account-tools');
			}
			else {
				// Check username
				if(!filter_var($request['login'], FILTER_VALIDATE_EMAIL)) {
					$status = 'blocked';
					$fields = array('login', 'password');
					$message = __('Incorrect username or password. Please correct your information and try again.', 'ejabberd-account-tools');
				}
				else {
					// Check host
					list($user, $host) = explode('@', $request['login']);
					if(false === ($transient = get_transient('ejabat_registered_vhosts'))) {
						$response = ejabat_get_xmpp_data('registered_vhosts');
						set_transient('ejabat_registered_vhosts', $response['body']);
						$transient = $response['body'];
					}
					if(false === array_search($host, json_decode($transient))) {
						$status = 'blocked';
						$fields = array('login', 'password');
						$message = __('Incorrect username or password. Please correct your information and try again.', 'ejabberd-account-tools');
					}
					else {
						// Check login and password
						$response = ejabat_get_xmpp_data('check_password', array('user' => $user, 'host' => $host, 'password' => $request['password']));
						// Server unavailable
						if(is_null($response)) {
							$status = 'error';
							$message = __('The server is temporarily unavailable. Please try again later.', 'ejabberd-account-tools');
						}
						// Invalid login or password
						else if($response['code'] == 1) {
							$status = 'blocked';
							$fields = array('login', 'password');
							$message = __('Incorrect username or password. Please correct your information and try again.', 'ejabberd-account-tools');
						}
						// Login and password valid
						else if($response['code'] == 0) {
							// Get current private email address
							$response = ejabat_get_xmpp_data('private_get', array('user' => $user, 'host' => $host, 'element' => 'private', 'ns' => 'email'));
							// Server unavailable
							if(is_null($response)) {
								$status = 'error';
								$message = __('The server is temporarily unavailable. Please try again later.', 'ejabberd-account-tools');
							}
							// Check response
							else if($response['code'] == 0) {
								// New email address different from current
								if($request['email'] != json_decode(strip_tags($response['body']))) {
									// Get current timestamp
									$now = time();
									// Set code transient
									$code = bin2hex(openssl_random_pseudo_bytes(16));
									$data = array('timestamp' => $now, 'ip' => $_SERVER['REMOTE_ADDR'], 'user' => $user, 'host' => $host, 'email' => $request['email']);
									set_transient('ejabat_email_'.$code, $data, get_option('ejabat_change_email_timeout', 900));
									// Email data
									$subject = sprintf(__('Confirm the email address for your account on %s', 'ejabberd-account-tools'), $host);
									$body = sprintf(__('Hey %s!<br><br>You have changed the private email address for your XMPP account %s. To complete the change, please click the following link:<br><br>%s<br><br>If you haven\'t made this change, simply ignore this email.<br><br>Best regards,<br>%s', 'ejabberd-account-tools'), ejabbat_get_vcard_name($user, $host), $user.'@'.$host, '<a href="'.explode('?', $_SERVER['HTTP_REFERER'])[0].'?code='.$code.'">'.explode('?', $_SERVER['HTTP_REFERER'])[0].'?code='.$code.'</a>', get_bloginfo('name'));
									$headers[] = 'From: '.get_bloginfo('name').' <'.get_option('admin_email').'>';
									$headers[] = 'Content-Type: text/html; charset=UTF-8';
									// Try send email
									if(wp_mail($user.' <'.$request['email'].'>', $subject, $body, $headers)) {
										// Email changes watcher
										if(get_option('ejabat_watch_email_changes', false) && get_option('ejabat_watcher')) {
											$watchers = explode(' ', get_option('ejabat_watcher'));
											foreach($watchers as $watcher) {
												ejabat_get_xmpp_data('send_message', array('type' => 'chat', 'from' => $host, 'to' => $watcher, 'subject' => '', 'body' => sprintf('[%s] User %s has requested from IP address %s to change the email address from %s to %s', wp_date('Y-m-d H:i:s', $now), $user.'@'.$host, $_SERVER['REMOTE_ADDR'], json_decode(strip_tags($response['body'])), $request['email'])));
											}
										}
										// Success message
										$status = 'success';
										$message = __('An email has been sent to you to complete your changes. It contains a link that you must click.', 'ejabberd-account-tools');
									}
									// Problem with sending email
									else {
										delete_transient('ejabat_email_'.$code);
										$status = 'error';
										$message = __('Failed to send the email. Please try again.', 'ejabberd-account-tools');
									}
								}
								// New private email address same as current
								else {
									$status = 'blocked';
									$fields = array('email');
									$message = __('The selected private email address is already set up for this account.', 'ejabberd-account-tools');
								}
							}
						}
					}
				}
			}
		}
	}
	// Return response
	if(get_option('ejabat_debug', false) == false) return rest_ensure_response(array('status' => isset($status) ? $status : 'error', 'message' => isset($message) ? $message : __('An unexpected error occurred. Please try again.', 'ejabberd-account-tools'), 'fields' => isset($fields) ? $fields : null));
	else return rest_ensure_response(array('status' => isset($status) ? $status : 'error', 'message' => isset($message) ? $message : __('An unexpected error occurred. Please try again.', 'ejabberd-account-tools'), 'fields' => isset($fields) ? $fields : null, 'debug_message' => isset($response['body']) ? $response['body'] : null, 'debug_code' => isset($response['code']) ? $response['code'] : null, 'debug_command' => isset($response['command']) ? $response['command'] : null, 'debug_arguments' => isset($response['arguments']) ? $response['arguments'] : null));
}
