<?php
/**
 * Form to delete account
 */

if(!defined('ABSPATH')) {
	exit;
}

// Enqueue styles & scripts
function ejabat_enqueue_delete_account_scripts() {
	global $post;
	if(is_a($post, 'WP_Post') && has_shortcode($post->post_content, 'ejabat_delete_account')) {
		wp_enqueue_style('ejabat', EJABAT_DIR_URL.'css/style.min.css', array(), EJABAT_VERSION, 'all');
		wp_enqueue_script('ejabat-form', EJABAT_DIR_URL.'js/js.ejabat.form.min.js', array(), EJABAT_VERSION, true);
		wp_localize_script('ejabat-form', 'ejabat', array(
			'nonce' => ejabat_create_nonce(),
			'rest_api' => esc_url_raw(rest_url().'ejabberd-account-tools/v1/'),
			'empty_field' => __('Please complete the required field.', 'ejabberd-account-tools'),
			'empty_fields' => __('Verification errors occurred. Please check all fields and submit the form again.', 'ejabberd-account-tools'),
			'error' => __('An unexpected error occurred. Please try again.', 'ejabberd-account-tools'),
			'form_error' => '<p class="ejabat"><span class="ejabat-info ejabat-error">'.__('An unexpected error occurred. Please try again.', 'ejabberd-account-tools').'</span></p>'
		));
	}
}
add_action('wp_enqueue_scripts', 'ejabat_enqueue_delete_account_scripts');

// Delete account shortcode
function ejabat_delete_account_shortcode() {
	return '<p data-action="delete-account-form" class="ejabat">'.(get_option('ejabat_loader', true) && true ? '<span class="ejabat-loader" title="'.__('Loading', 'ejabberd-account-tools').'..."></span>' : '').'</p>';
}
add_shortcode('ejabat_delete_account', 'ejabat_delete_account_shortcode');

// Route delete account form
function ejabat_route_delete_account_form() {
	register_rest_route('ejabberd-account-tools/v1', '/delete-account-form', array(
		'methods' => 'POST',
		'callback' => 'ejabat_delete_account_form',
		'permission_callback' => '__return_true',
		'args' => array(
			'code' => array(
				'type' => 'string',
				'default' => 'undefined',
				'sanitize_callback' => function($param, $request, $key) {
					return stripslashes_deep(sanitize_text_field($param));
				}
			)
		)
	));
}
add_action('rest_api_init', 'ejabat_route_delete_account_form');

// Delete account form
function ejabat_delete_account_form($request) {
	// Form is disabled
	if(get_option('ejabat_disable_delete_account', false) && !is_user_logged_in()) {
		$html = '<p class="ejabat"><span class="ejabat-info ejabat-error">'.__('The form to delete account is temporarily disabled. Please try again later.', 'ejabberd-account-tools').'</span></p>';
	}
	else {
		// Link to delete account
		if($request['code'] != 'undefined') {
			// Code valid
			if(true == ($transient = get_transient('ejabat_unreg_'.$request['code']))) {
				// Create form
				$html = '<form data-action="unregister-account" class="ejabat" method="post" onsubmit="return false" autocomplete="off">
					<p class="ejabat-info ejabat-error">'.__('If you enter your correct password here, your account will be deleted forever. It will not be possible to restore it.', 'ejabberd-account-tools').'</p>
					<p id="login">
						<input type="text" name="login" value="'.$transient['user'].'@'.$transient['host'].'" disabled>
						<span class="ejabat-tip"></span>
					</p>
					<p id="password" class="ejabat-validate">
						<input type="password" name="password" placeholder="'.__('Password', 'ejabberd-account-tools').'">
						<span class="ejabat-tip"></span>
					</p>
					<p>
						<input type="submit" value="'.__('Yes, really delete my account', 'ejabberd-account-tools').'">
						<input type="hidden" name="code" value="'.$request['code'].'">
						<span class="ejabat-spinner" style="visibility: hidden;"></span>
					</p>
					<div id="response"></div>
				</form>';
			}
			// Code expired or not valid
			else {
				// Delete transient
				delete_transient('ejabat_unreg_'.$request['code']);
				// Response with error
				$message = '<p id="message" class="ejabat-info ejabat-blocked">'.__('The link to delete your account has expired or is invalid. Please complete the form and resubmit it.', 'ejabberd-account-tools').'</p>';
			}
		}
		// Create form
		$html = isset($html) ? $html : '<form data-action="delete-account" class="ejabat" method="post" onsubmit="return false" autocomplete="off">
			'.(isset($message) ? $message : '').'
			<p id="login" class="ejabat-validate">
				<input type="text" name="login" placeholder="'.__('Full username', 'ejabberd-account-tools').'">
				<span class="ejabat-tip"></span>
			</p>
			<p id="password" class="ejabat-validate">
				<input type="password" name="password" placeholder="'.__('Password', 'ejabberd-account-tools').'">
				<span class="ejabat-tip"></span>
			</p>
			'.ejabat_captcha_field().'
			<p class="ejabat-info ejabat-error">'.__('If you delete your account, it\'s gone forever. There is no way to restore it.', 'ejabberd-account-tools').'</p>
			<p>
				<input type="submit" value="'.__('Delete account', 'ejabberd-account-tools').'">
				<span class="ejabat-spinner" style="visibility: hidden;"></span>
			</p>
			<div id="response"></div>
		</form>';
	}
	return rest_ensure_response(array('data' => str_replace(array(PHP_EOL, "\t"), '', $html), 'nonce' => wp_create_nonce('wp_rest')));
}

// Route delete account
function ejabat_route_delete_account() {
	register_rest_route('ejabberd-account-tools/v1', '/delete-account', array(
		'methods' => 'POST',
		'callback' => 'ejabat_delete_account',
		'permission_callback' => '__return_true',
		'args' => array(
			'login' => array(
				'type' => 'string',
				'required' => true,
				'sanitize_callback' => function($param, $request, $key) {
					return stripslashes_deep(sanitize_text_field($param));
				}
			),
			'password' => array(
				'type' => 'string',
				'required' => true,
				'sanitize_callback' => function($param, $request, $key) {
					return stripslashes_deep(sanitize_text_field($param));
				}
			)
		)
	));
}
add_action('rest_api_init', 'ejabat_route_delete_account');

// Delete account callback
function ejabat_delete_account($request) {
	// Verify nonce
	if(!wp_verify_nonce($request->get_header('x-wp-nonce'), 'wp_rest')) {
		$status = 'blocked';
		$message = __('Verification error. Please try again.', 'ejabberd-account-tools');
	}
	else {
		// Verify captcha
		if(true !== ($captcha_verify = ejabat_captcha_verify($request))) {
			$status = 'blocked';
			if(is_array($captcha_verify)) $fields = $captcha_verify;
			$message = __('Captcha validation error. Please try again.', 'ejabberd-account-tools');
		}
		else {
			// Check username
			if(!filter_var($request['login'], FILTER_VALIDATE_EMAIL)) {
				$status = 'blocked';
				$fields = array('login', 'password');
				$message = __('Incorrect username or password. Please correct your information and try again.', 'ejabberd-account-tools');
			}
			else {
				// Check host
				list($user, $host) = explode('@', $request['login']);
				if(false === ($transient = get_transient('ejabat_registered_vhosts'))) {
					$response = ejabat_get_xmpp_data('registered_vhosts');
					set_transient('ejabat_registered_vhosts', $response['body']);
					$transient = $response['body'];
				}
				if(false === array_search($host, json_decode($transient))) {
					$status = 'blocked';
					$fields = array('login', 'password');
					$message = __('Incorrect username or password. Please correct your information and try again.', 'ejabberd-account-tools');
				}
				else {
					// Check login and password
					$response = ejabat_get_xmpp_data('check_password', array('user' => $user, 'host' => $host, 'password' => $request['password']));
					// Server unavailable
					if(is_null($response)) {
						$status = 'error';
						$message = __('The server is temporarily unavailable. Please try again later.', 'ejabberd-account-tools');
					}
					// Invalid login or password
					else if($response['code'] == 1) {
						$status = 'blocked';
						$fields = array('login', 'password');
						$message = __('Incorrect username or password. Please correct your information and try again.', 'ejabberd-account-tools');
					}
					// Login and password valid
					else if($response['code'] == 0) {
						// Get private email address
						$response = ejabat_get_xmpp_data('private_get', array('user' => $user, 'host' => $host, 'element' => 'private', 'ns' => 'email'));
						// Server unavailable
						if(is_null($response)) {
							$status = 'error';
							$message = __('The server is temporarily unavailable. Please try again later.', 'ejabberd-account-tools');
						}
						// Check response
						else if($response['code'] == 0) {
							// Private email set
							if(true == ($email = json_decode(strip_tags($response['body'])))) {
								// Get current timestamp
								$now = time();
								// Set code transient
								$code = bin2hex(openssl_random_pseudo_bytes(16));
								$data = array('timestamp' => $now, 'ip' => $_SERVER['REMOTE_ADDR'], 'user' => $user, 'host' => $host, 'email' => $email);
								set_transient('ejabat_unreg_'.$code, $data, get_option('ejabat_delete_account_timeout', 900));
								// Email data
								$subject = sprintf(__('Confirm deletion of your account on %s', 'ejabberd-account-tools'), $host);
								$body = sprintf(__('Hey %s!<br><br>You wish to delete your XMPP account %s. To complete the change, please click the following link:<br><br>%s<br><br>If you no longer wish to delete your account, simply ignore this email.<br><br>Best regards,<br>%s', 'ejabberd-account-tools'), ejabbat_get_vcard_name($user, $host), $user.'@'.$host, '<a href="'.explode('?', $_SERVER['HTTP_REFERER'])[0].'?code='.$code.'">'.explode('?', $_SERVER['HTTP_REFERER'])[0].'?code='.$code.'</a>', get_bloginfo('name'));
								$headers[] = 'From: '.get_bloginfo('name').' <'.get_option('admin_email').'>';
								$headers[] = 'Content-Type: text/html; charset=UTF-8';
								// Try send email
								if(wp_mail($user.' <'.$email.'>', $subject, $body, $headers)) {
									// Delete account watcher
									if(get_option('ejabat_watch_delete_account', false) && get_option('ejabat_watcher')) {
										$watchers = explode(' ', get_option('ejabat_watcher'));
										foreach($watchers as $watcher) {
											ejabat_get_xmpp_data('send_message', array('type' => 'chat', 'from' => $host, 'to' => $watcher, 'subject' => '', 'body' => sprintf('[%s] User %s has requested from IP address %s to delete the account', wp_date('Y-m-d H:i:s', $now), $user.'@'.$host, $_SERVER['REMOTE_ADDR'])));
										}
									}
									$status = 'success';
									$message = sprintf(__('An email has been sent to you at %s. It contains a link to a page where you can delete your account.', 'ejabberd-account-tools'), mask_email($email));
								}
								// Problem with sending email
								else {
									// Delete code transient
									delete_transient('ejabat_unreg_'.$code);
									// Error message
									$status = 'error';
									$message = __('Failed to send the email. Please try again.', 'ejabberd-account-tools');
								}
							}
							// Private email not set
							else {
								$status = 'blocked';
								$message = __('A private email address hasn\'t been set. To delete the account, set a private email address or simply delete account via IM.', 'ejabberd-account-tools');
							}
						}
					}
				}
			}
		}
	}
	// Return response
	if(get_option('ejabat_debug', false) == false) return rest_ensure_response(array('status' => isset($status) ? $status : 'error', 'message' => isset($message) ? $message : __('An unexpected error occurred. Please try again.', 'ejabberd-account-tools'), 'fields' => isset($fields) ? $fields : null));
	else return rest_ensure_response(array('status' => isset($status) ? $status : 'error', 'message' => isset($message) ? $message : __('An unexpected error occurred. Please try again.', 'ejabberd-account-tools'), 'fields' => isset($fields) ? $fields : null, 'debug_message' => isset($response['body']) ? $response['body'] : null, 'debug_code' => isset($response['code']) ? $response['code'] : null, 'debug_command' => isset($response['command']) ? $response['command'] : null, 'debug_arguments' => isset($response['arguments']) ? $response['arguments'] : null));
}

// Route unregister account
function ejabat_route_unregister_account() {
	register_rest_route('ejabberd-account-tools/v1', '/unregister-account', array(
		'methods' => 'POST',
		'callback' => 'ejabat_unregister_account',
		'permission_callback' => '__return_true',
		'args' => array(
			'code' => array(
				'type' => 'string',
				'required' => true,
				'sanitize_callback' => function($param, $request, $key) {
					return stripslashes_deep(sanitize_text_field($param));
				}
			),
			'password' => array(
				'type' => 'string',
				'required' => true,
				'sanitize_callback' => function($param, $request, $key) {
					return stripslashes_deep(sanitize_text_field($param));
				}
			)
		)
	));
}
add_action('rest_api_init', 'ejabat_route_unregister_account');

// Unregister account callback
function ejabat_unregister_account($request) {
	// Verify nonce
	if(!wp_verify_nonce($request->get_header('x-wp-nonce'), 'wp_rest')) {
		$status = 'blocked';
		$message = __('Verification error. Please try again.', 'ejabberd-account-tools');
	}
	else {
		// Code valid
		if(true == ($transient = get_transient('ejabat_unreg_'.$request['code']))) {
			// Check login and password
			$response = ejabat_get_xmpp_data('check_password', array('user' => $transient['user'], 'host' => $transient['host'], 'password' => $request['password']));
			// Server unavailable
			if(is_null($response)) {
				$status = 'error';
				$message = __('The server is temporarily unavailable. Please try again later.', 'ejabberd-account-tools');
			}
			// Invalid password
			else if($response['code'] == 1) {
				$status = 'blocked';
				$fields = array('password');
				$message = __('Incorrect password. Please correct it and try again.', 'ejabberd-account-tools');
			}
			// Login and password valid
			else if($response['code'] == 0) {
				// Try to unregister account
				$response = ejabat_get_xmpp_data('unregister', array('user' => $transient['user'], 'host' => $transient['host']));
				// Server unavailable
				if(is_null($response)) {
					$status = 'error';
					$message = __('The server is temporarily unavailable. Please try again later.', 'ejabberd-account-tools');
				}
				// Account unregistered
				else if($response['code'] == 0) {
					// Delete account watcher
					if(get_option('ejabat_watch_delete_account', false) && get_option('ejabat_watcher')) {
						$now = time();
						$watchers = explode(' ', get_option('ejabat_watcher'));
						foreach($watchers as $watcher) {
							ejabat_get_xmpp_data('send_message', array('type' => 'chat', 'from' => $transient['host'], 'to' => $watcher, 'subject' => '', 'body' => sprintf('[%s] User %s has deleted the account from IP address %s', wp_date('Y-m-d H:i:s', $now), $transient['user'].'@'.$transient['host'], $_SERVER['REMOTE_ADDR'])));
						}
					}
					// Delete code transient
					delete_transient('ejabat_unreg_'.$request['code']);
					// Success message
					$status = 'success';
					$message = __('Thank you for using our service. Your account has been deleted.', 'ejabberd-account-tools');
				}
			}
		}
		// Code expired or not valid
		else {
			// Delete transient
			delete_transient('ejabat_pass_'.$request['code']);
			// Error message
			$status = 'blocked';
			$message = __('The link to delete your account has expired or is invalid.', 'ejabberd-account-tools');
		}
	}
	// Return response
	if(get_option('ejabat_debug', false) == false) return rest_ensure_response(array('status' => isset($status) ? $status : 'error', 'message' => isset($message) ? $message : __('An unexpected error occurred. Please try again.', 'ejabberd-account-tools'), 'fields' => isset($fields) ? $fields : null));
	else return rest_ensure_response(array('status' => isset($status) ? $status : 'error', 'message' => isset($message) ? $message : __('An unexpected error occurred. Please try again.', 'ejabberd-account-tools'), 'fields' => isset($fields) ? $fields : null, 'debug_message' => isset($response['body']) ? $response['body'] : null, 'debug_code' => isset($response['code']) ? $response['code'] : null, 'debug_command' => isset($response['command']) ? $response['command'] : null, 'debug_arguments' => isset($response['arguments']) ? $response['arguments'] : null));
}
