<?php
/**
 * The most important functions used in other files of the plugin
 */

if(!defined('ABSPATH')) {
	exit;
}

// POST request to the XMPP server
function ejabat_get_xmpp_data($command, $arguments = array()) {
	// Perform request
	$response = wp_remote_post(get_option('ejabat_rest_url').'/'.$command, array(
		'headers' => array(
			'Authorization' => empty(get_option('ejabat_oauth_token')) ? 'Basic '.base64_encode(get_option('ejabat_login').':'.get_option('ejabat_password')) : 'Bearer '.get_option('ejabat_oauth_token'),
			'Content-Type' => 'application/json'
		),
		'body' => json_encode($arguments, JSON_FORCE_OBJECT),
		'redirection' => 0,
		'httpversion' => '1.1'
	));
	// Server unavailable
	if(is_wp_error($response)) {
		return null;
	}
	// Verify response
	else if($response['response']['code'] == 200) {
		if(is_numeric($response['body'])) return array('code' => $response['body'], 'body' => $response['body'], 'command' => $command, 'arguments' => $arguments);
		else return array('code' => '0', 'body' => $response['body'], 'command' => $command, 'arguments' => $arguments);
	}
	// Unexpected error
	return array('code' => $response['response']['code'], 'body' => $response['response']['message'], 'command' => $command, 'arguments' => $arguments);
}

// Validating email address by checking MX record
function ejabat_validate_email_mxrecord($email) {
	// Explode email
	list($user, $domain) = explode('@', $email);
	// Check MX record
	return checkdnsrr($domain, 'MX');
}

// Validating email address on usercheck.com
function ejabat_validate_email_usercheck_com($email) {
	if(get_option('ejabat_validator_pizza', true)) {
		// Explode email
		list($user, $domain) = explode('@', $email);
		// Get disposable email status from cache
		if(true == ($transient = get_transient('ejabat_usercheck_com_'.$domain))) {
			return $transient;
		}
		else {
			// GET arguments
			$args = array(
				'redirection' => 0,
				'httpversion' => '1.1'
			);
			// GET data
			$response = wp_remote_get('https://api.usercheck.com/domain/' . $domain, $args);
			// Server temporarily unavailable
			if(is_wp_error($response)) {
				return true;
			}
			// Verify response
			else if($response['response']['code'] == 200) {
				// Save disposable email status to cache
				set_transient('ejabat_usercheck_com_'.$domain, !boolval(json_decode($response['body'])->disposable));
				// Return response body
				return !boolval(json_decode($response['body'])->disposable);
			}
		}
	}
	// Unexpected error
	return true;
}

// Route validate email address
function ejabat_route_validate_email() {
	register_rest_route('ejabberd-account-tools/v1', '/validate-email', array(
		'methods' => 'POST',
		'callback' => 'ejabat_validate_email',
		'permission_callback' => '__return_true',
		'args' => array(
			'email' => array(
				'type' => 'string',
				'required' => true,
				'sanitize_callback' => function($param, $request, $key) {
					return stripslashes_deep(sanitize_text_field($param));
				}
			)
		)
	));
}
add_action('rest_api_init', 'ejabat_route_validate_email');

// Validate email address
function ejabat_validate_email($request) {
	// Verify email
	if(!filter_var($request['email'], FILTER_VALIDATE_EMAIL) || !ejabat_validate_email_mxrecord($request['email'])) {
		$status = 'blocked';
	}
	// Return response
	return rest_ensure_response(array('status' => isset($status) ? $status : 'success'));
}

// Masking email address
function mask_email($email, $mask = '*', $percent = 80) {
	list($user, $domain) = explode('@', $email);
	$len = strlen($user);
	$mask_count = floor($len * $percent / 100);
	$offset = floor(($len - $mask_count) / 2);
	$masked = substr($user, 0, $offset).str_repeat($mask, $mask_count).substr($user, $mask_count + $offset);
	return($masked.'@'.$domain);
}

// Route checking account
function ejabat_route_check_account() {
	register_rest_route('ejabberd-account-tools/v1', '/check-account', array(
		'methods' => 'POST',
		'callback' => 'ejabat_check_account',
		'permission_callback' => '__return_true',
		'args' => array(
			'username' => array(
				'type' => 'string',
				'required' => true,
				'sanitize_callback' => function($param, $request, $key) {
					return stripslashes_deep(sanitize_text_field($param));
				}
			),
			'host' => array(
				'type' => 'string',
				'required' => true,
				'sanitize_callback' => function($param, $request, $key) {
					return stripslashes_deep(sanitize_text_field($param));
				}
			)
		)
	));
}
add_action('rest_api_init', 'ejabat_route_check_account');

// Checking account
function ejabat_check_account($request) {
	// Verify username
	if(!preg_match('/'.get_option('ejabat_allowed_username_regexp', '^(?=^[a-z0-9].*)(?=.*[a-z0-9]$)(?!.*([_.-])\1+.*)(?!^([0-9]+)$)[a-z0-9_.-]{3,30}$').'/i', $request['username'])) {
		$status = 'blocked';
		$message = __('Username doesn\'t meet the requirements.', 'ejabberd-account-tools');
	}
	else if(preg_match('/'.get_option('ejabat_blocked_username_regexp', '^(.*(404|abort|about|abuse|access|account|activat|address|adium|admin|adult|advertisin|affiliat|agile|ajax|allegro|analytics|android|anonymous|api|app|aqq|archiv|atom|auth|backup|billing|blog|board|bombus|bot|bug|business|cache|calendar|campaign|cancel|careers|cart|ceo|cgi|changelog|chat|check|chrome|client|cms|comercial|comment|compare|config|connect|contact|contest|contract|convers|cpp|creat|css|custome|dashboard|delete|demo|design|detail|develop|digsby|direct|disc|docs|document|domain|dot|drive|dropbox|ebay|ecommerce|edit|employment|enquiries|enterprise|error|event|facebook|faq|favorite|feed|file|firefox|flock|follow|form|forum|ftp|gadget|gajim|gist|github|google|group|guest|guide|help|homepage|host|htm|http|ijab|imap|index|info|instagram|instantbird|internal|intranet|invit|invoic|ipad|iphone|irc|irssi|issue|jabbear|jabber|jabbim|jabiru|jappix|java|jitsi|job|joomla|json|kadu|kopete|language|load|login|logout|logs|mail|manager|manual|market|media|member|message|messenger|microblog|microsoft|miranda|mobile|mozilla|mp3|msg|msn|mysql|name|network|news|nick|noreply|ns1|ns2|ns3|ns4|oauth|offers|office|olx|online|openid|operator|oracle|order|organizat|owner|page|pandion|panel|password|perl|php|pidgin|plugin|pop3|popular|porn|post|press|print|privacy|profil|promo|psi|pub|python|query|random|recruit|register|registrat|remove|replies|root|rss|ruby|sales|sample|save|script|search|secure|security|send|seo|service|session|setting|setup|shop|signin|signup|site|smtp|sql|ssh|ssl|staff|start|static|stats|status|store|subscrib|support|sysop|system|tablet|talk|task|team|tech|telnet|terms|test|theme|tigase|tkabber|tlen|tmp|todo|tool|translat|trillian|troll|tube|twitt|update|url|usage|user|vendas|video|visitor|voice|weather|web|widget|windows|work|wtw|www|xabber|xml|xmpp|yaml|yaxim|yml).*)$').'/i', $request['username'])) {
		$status = 'blocked';
		$message = __('Username contains forbidden words.', 'ejabberd-account-tools');
	}
	// Check account
	else {
		// Check host
		$hosts = explode(' ', get_option('ejabat_registration_hosts', preg_replace('/^www\./','',$_SERVER['SERVER_NAME'])));
		if(in_array($request['host'], $hosts)) {
			// Check account
			$response = ejabat_get_xmpp_data('check_account', array('user' => $request['username'], 'host' => $request['host']));
			// Server unavailable
			if(is_null($response)) {
				$status = 'error';
				$message = __('The server is temporarily unavailable. Please try again later.', 'ejabberd-account-tools');
			}
			// Username available
			else if($response['code'] == 1) {
				$status = 'success';
				$message = __('The selected username is available.', 'ejabberd-account-tools');
			}
			// Username already registered
			else if($response['code'] == 0) {
				$status = 'blocked';
				$message = __('The selected username is already registered.', 'ejabberd-account-tools');
			}
		}
		// Invalid host
		else {
			$status = 'error';
			$message = __('Invalid hostname. Please try again.', 'ejabberd-account-tools');
		}
	}
	// Return response
	if(get_option('ejabat_debug', false) == false) return rest_ensure_response(array('status' => isset($status) ? $status : 'error', 'message' => isset($message) ? $message : __('An unexpected error occurred. Please try again.', 'ejabberd-account-tools')));
	else return rest_ensure_response(array('status' => isset($status) ? $status : 'error', 'message' => isset($message) ? $message : __('An unexpected error occurred. Please try again.', 'ejabberd-account-tools'), 'debug_message' => isset($response['body']) ? $response['body'] : null, 'debug_code' => isset($response['code']) ? $response['code'] : null, 'debug_command' => isset($response['command']) ? $response['command'] : null, 'debug_arguments' => isset($response['arguments']) ? $response['arguments'] : null));
}

// Route login
function ejabat_route_login() {
	register_rest_route('ejabberd-account-tools/v1', '/login', array(
		'methods' => 'POST',
		'callback' => 'ejabat_login',
		'permission_callback' => '__return_true',
		'args' => array(
			'login' => array(
				'type' => 'string',
				'required' => true,
				'sanitize_callback' => function($param, $request, $key) {
					return stripslashes_deep(sanitize_text_field($param));
				}
			),
			'password' => array(
				'type' => 'string',
				'required' => true,
				'sanitize_callback' => function($param, $request, $key) {
					return stripslashes_deep(sanitize_text_field($param));
				}
			)
		)
	));
}
add_action('rest_api_init', 'ejabat_route_login');

// Login callback
function ejabat_login($request) {
	// Verify captcha
	if(true !== ($captcha_verify = ejabat_captcha_verify($request))) {
		$status = 'blocked';
		if(is_array($captcha_verify)) $fields = $captcha_verify;
		$message = __('Captcha validation error. Please try again.', 'ejabberd-account-tools');
	}
	else {
		// Check username
		if(!filter_var($request['login'], FILTER_VALIDATE_EMAIL)) {
			$status = 'blocked';
			$fields = array('login', 'password');
			$message = __('Incorrect username or password. Please correct your information and try again.', 'ejabberd-account-tools');
		}
		else {
			// Check host
			list($user, $host) = explode('@', $request['login']);
			if(false === ($transient = get_transient('ejabat_registered_vhosts'))) {
				$response = ejabat_get_xmpp_data('registered_vhosts');
				set_transient('ejabat_registered_vhosts', $response['body']);
				$transient = $response['body'];
			}
			if(false === array_search($host, json_decode($transient))) {
				$status = 'blocked';
				$fields = array('login', 'password');
				$message = __('Incorrect username or password. Please correct your information and try again.', 'ejabberd-account-tools');
			}
			else {
				// Check login and password
				$response = ejabat_get_xmpp_data('check_password', array('user' => $user, 'host' => $host, 'password' => $request['password']));
				// Server unavailable
				if(is_null($response)) {
					$status = 'error';
					$message = __('The server is temporarily unavailable. Please try again later.', 'ejabberd-account-tools');
				}
				// Invalid login or password
				else if($response['code'] == 1) {
					$status = 'blocked';
					$fields = array('login', 'password');
					$message = __('Incorrect username or password. Please correct your information and try again.', 'ejabberd-account-tools');
				}
				// Login and password valid
				else if($response['code'] == 0) {
					// Set session transient
					$session = bin2hex(openssl_random_pseudo_bytes(16));
					$data = array('timestamp' => time(), 'ip' => $_SERVER['REMOTE_ADDR'], 'user' => $user, 'host' => $host);
					// New session successfully created
					if(set_transient('ejabat_session_'.$session, $data, get_option('ejabat_delete_account_timeout', 1800))) {
						$status = 'success';
						$message = __('Logged into account, redirecting...', 'ejabberd-account-tools');
					}
				}
			}
		}
	}
	// Return response
	if(get_option('ejabat_debug', false) == false) return rest_ensure_response(array('status' => isset($status) ? $status : 'error', 'message' => isset($message) ? $message : __('An unexpected error occurred. Please try again.', 'ejabberd-account-tools'), 'fields' => isset($fields) ? $fields : null, 'session' => $session ?: 'expired'));
	else return rest_ensure_response(array('status' => isset($status) ? $status : 'error', 'message' => isset($message) ? $message : __('An unexpected error occurred. Please try again.', 'ejabberd-account-tools'), 'fields' => isset($fields) ? $fields : null, 'session' => $session ?: 'expired', 'debug_message' => isset($response['body']) ? $response['body'] : null, 'debug_code' => isset($response['code']) ? $response['code'] : null, 'debug_command' => isset($response['command']) ? $response['command'] : null, 'debug_arguments' => isset($response['arguments']) ? $response['arguments'] : null));
}

// Route logout
function ejabat_route_logout() {
	register_rest_route('ejabberd-account-tools/v1', '/logout', array(
		'methods' => 'POST',
		'callback' => 'ejabat_logout',
		'permission_callback' => '__return_true',
		'args' => array(
			'session' => array(
				'type' => 'string',
				'required' => true,
				'sanitize_callback' => function($param, $request, $key) {
					return stripslashes_deep(sanitize_text_field($param));
				}
			)
		)
	));
}
add_action('rest_api_init', 'ejabat_route_logout');

// Logout callback
function ejabat_logout($request) {
	// Delete transient
	if(delete_transient('ejabat_session_'.$request['session'])) {
		$status = 'success';
		$message = 'expired';
	}
	// Return response
	if(get_option('ejabat_debug', false) == false) return rest_ensure_response(array('status' => isset($status) ? $status : 'error', 'message' => isset($message) ? $message : __('An unexpected error occurred. Please try again.', 'ejabberd-account-tools')));
	else return rest_ensure_response(array('status' => isset($status) ? $status : 'error', 'message' => isset($message) ? $message : __('An unexpected error occurred. Please try again.', 'ejabberd-account-tools'), 'debug_message' => isset($response['body']) ? $response['body'] : null, 'debug_code' => isset($response['code']) ? $response['code'] : null, 'debug_command' => isset($response['command']) ? $response['command'] : null, 'debug_arguments' => isset($response['arguments']) ? $response['arguments'] : null));
}

// Get name from vCard
function ejabbat_get_vcard_name($user, $host) {
	// Get nickname
	$response = ejabat_get_xmpp_data('get_vcard', array('user' => $user, 'host' => $host, 'name' => 'NICKNAME'));
	// Nickname is set
	if($response['code'] == 0) {
		return json_decode($response['body']);
	}
	else {
		// Get full name
		$response = ejabat_get_xmpp_data('get_vcard', array('user' => $user, 'host' => $host, 'name' => 'FN'));
		// Full name is set
		if($response['code'] == 0) {
			return json_decode($response['body']);
		}
		else return $user;
	}
}

// Create nonce for logged in user
function ejabat_create_nonce() {
	return is_user_logged_in() ? wp_create_nonce('wp_rest') : '';
}
