<?php
/**
 * Registration form
 */

if(!defined('ABSPATH')) {
	exit;
}

// Enqueue styles & scripts
function ejabat_enqueue_register_scripts() {
	global $post;
	if(is_a($post, 'WP_Post') && has_shortcode($post->post_content, 'ejabat_register')) {
		wp_enqueue_style('ejabat', EJABAT_DIR_URL.'css/style.min.css', array(), EJABAT_VERSION, 'all');
		wp_enqueue_script('ejabat-form', EJABAT_DIR_URL.'js/js.ejabat.form.min.js', array(), EJABAT_VERSION, true);
		if(get_option('ejabat_show_hints', true)) {
			$hints = apply_filters('ejabat_hints_args', array(
				'username' => get_option('ejabat_username_hint', __('At least 3 and up to 30 characters, only letters and numbers.', 'ejabberd-account-tools')),
				'password' => get_option('ejabat_password_hint', __('At least a good password is required.', 'ejabberd-account-tools')),
				'email' => get_option('ejabat_email_hint', __('Only required for password recovery.', 'ejabberd-account-tools'))
			));
		}
		wp_localize_script('ejabat-form', 'ejabat', array(
			'nonce' => ejabat_create_nonce(),
			'rest_api' => esc_url_raw(rest_url().'ejabberd-account-tools/v1/'),
			'username_regexp' => get_option('ejabat_allowed_username_regexp', '^(?=^[a-z0-9].*)(?=.*[a-z0-9]$)(?!.*([_.-])\1+.*)(?!^([0-9]+)$)[a-z0-9_.-]{3,30}$'),
			'checking_username' => '<span class="ejabat-spinner"></span>'.__('Checking username...', 'ejabberd-account-tools'),
			'invalid_username' => __('Username doesn\'t meet the requirements.', 'ejabberd-account-tools'),
			'password_strength' => get_option('ejabat_password_strength', 3),
			'password_too_weak' => __('The password is too weak.', 'ejabberd-account-tools'),
			'password_very_weak' => __('The password is very weak.', 'ejabberd-account-tools'),
			'password_weak' => __('The password is weak.', 'ejabberd-account-tools'),
			'password_good' => __('The password is good.', 'ejabberd-account-tools'),
			'password_strong' => __('The password is strong.', 'ejabberd-account-tools'),
			'checking_email' => '<span class="ejabat-spinner"></span>'.__('Checking email address...', 'ejabberd-account-tools'),
			'invalid_email' => __('The email address is invalid.', 'ejabberd-account-tools'),
			'did_you_mean' => __('Did you mean', 'ejabberd-account-tools'),
			'empty_field' => __('Please complete the required field.', 'ejabberd-account-tools'),
			'empty_fields' => __('Verification errors occurred. Please check all fields and submit the form again.', 'ejabberd-account-tools'),
			'username_hint' => $hints['username'],
			'password_hint' => $hints['password'],
			'email_hint' => $hints['email'],
			'error' => __('An unexpected error occurred. Please try again.', 'ejabberd-account-tools'),
			'form_error' => '<p class="ejabat"><span class="ejabat-info ejabat-error">'.__('An unexpected error occurred. Please try again.', 'ejabberd-account-tools').'</span></p>'
		));
		wp_enqueue_script('zxcvbn-async');
	}
}
add_action('wp_enqueue_scripts', 'ejabat_enqueue_register_scripts');

// Registration form shortcode
function ejabat_register_shortcode() {
	return '<p data-action="registration-form" class="ejabat">'.(get_option('ejabat_loader', true) && true ? '<span class="ejabat-loader" title="'.__('Loading', 'ejabberd-account-tools').'..."></span>' : '').'</p>';
}
add_shortcode('ejabat_register', 'ejabat_register_shortcode');

// Route registration form
function ejabat_route_registration_form() {
	register_rest_route('ejabberd-account-tools/v1', '/registration-form', array(
		'methods' => 'POST',
		'callback' => 'ejabat_registration_form',
		'permission_callback' => '__return_true',
		'args' => array(
			'host' => array(
				'type' => 'string',
				'default' => 'undefined',
				'sanitize_callback' => function($param, $request, $key) {
					return stripslashes_deep(sanitize_text_field($param));
				}
			),
			'code' => array(
				'type' => 'string',
				'default' => 'undefined',
				'sanitize_callback' => function($param, $request, $key) {
					return stripslashes_deep(sanitize_text_field($param));
				}
			)
		)
	));
}
add_action('rest_api_init', 'ejabat_route_registration_form');

// Registration form
function ejabat_registration_form($request) {
	// Registration is disabled
	if(get_option('ejabat_disable_registration', false) && !is_user_logged_in()) {
		$html = '<p class="ejabat"><span class="ejabat-info ejabat-error">'.__('Registration is temporarily disabled. Please try again later.', 'ejabberd-account-tools').'</span></p>';
	}
	else {
		// Verify registration timeout
		if(true == get_transient('ejabat_register_'.$_SERVER['REMOTE_ADDR'])) {
			$message = '<p id="message" class="ejabat-info ejabat-blocked">'.__('You can\'t register another account so quickly. Please try again later.', 'ejabberd-account-tools').'</p>';
		}
		// Link to confirm registration
		if($request['code'] != 'undefined') {
			// Code valid
			if(true == ($transient = get_transient('ejabat_register_'.$request['code']))) {
				// Verify required activation
				$response = ejabat_get_xmpp_data('get_ban_details', array('user' => $transient['username'], 'host' => $transient['host']));
				if((isset(json_decode($response['body'])->reason) ? json_decode($response['body'])->reason : '') == 'Activation required') {
					// Try set correct password by unban account
					$response = ejabat_get_xmpp_data('unban_account', array('user' => $transient['username'], 'host' => $transient['host']));
					// Server unavailable
					if(is_null($response)) {
						$html = '<p class="ejabat"><span class="ejabat-info ejabat-error">'.__('Your account activation failed because the server is temporarily unavailable. Please try again later.', 'ejabberd-account-tools').'</span></p>';
					}
					// Password changed
					else if($response['code'] == 0) {
						// Send welcome message
						if(get_option('ejabat_welcome_msg', false)) {
							// Get subject and body
							$welcome_msg = apply_filters('ejabat_welcome_msg_args', array(
								'subject' => get_option('ejabat_welcome_msg_subject'),
								'body' => get_option('ejabat_welcome_msg_body')
							));
							$welcome_msg['subject'] = htmlspecialchars(wp_strip_all_tags(do_shortcode($welcome_msg['subject'])));
							$welcome_msg['subject'] = str_replace(array('\r\n', '\\r\\n', '\n', '\\n', '\r', '\\r'), ' ', $welcome_msg['subject']);
							$welcome_msg['body'] = htmlspecialchars(wp_strip_all_tags(do_shortcode($welcome_msg['body'])));
							$welcome_msg['body'] = str_replace(array('\r\n', '\\r\\n', '\n', '\\n', '\r', '\\r'), "\n", $welcome_msg['body']);
							// Send message
							if($welcome_msg['subject']) {
								ejabat_get_xmpp_data('send_message', array('type' => 'normal', 'from' => $transient['host'], 'to' => $transient['username'].'@'.$transient['host'], 'subject' => $welcome_msg['subject'], 'body' => $welcome_msg['body']));
							} else {
								ejabat_get_xmpp_data('send_message', array('type' => 'chat', 'from' => $transient['host'], 'to' => $transient['username'].'@'.$transient['host'], 'body' => $welcome_msg['body']));
							}
						}
						// Activation watcher
						if(get_option('ejabat_watch_activation', false) && get_option('ejabat_watcher')) {
							$now = time();
							$watchers = explode(' ', get_option('ejabat_watcher'));
							foreach($watchers as $watcher) {
								ejabat_get_xmpp_data('send_message', array('type' => 'chat', 'from' => $transient['host'], 'to' => $watcher, 'subject' => '', 'body' => sprintf('[%s] Account %s was activated from IP address %s', wp_date('Y-m-d H:i:s', $now), $transient['username'].'@'.$transient['host'], $_SERVER['REMOTE_ADDR'])));
							}
						}
						// Delete transient
						delete_transient('ejabat_register_'.$request['code']);
						// Success message
						$html = '<p class="ejabat"><span class="ejabat-info ejabat-success">'.sprintf(__('Your account %s has been successfully activated.', 'ejabberd-account-tools'),$transient['username'].'@'.$transient['host']).'</span></p>';
					}
					// Unexpected error
					else {
						$html = '<p class="ejabat"><span class="ejabat-info ejabat-error">'.__('An unexpected error occurred while trying to activate your account. Please try again.', 'ejabberd-account-tools').'</span></p>';
					}
				}
				// Account already activated
				else {
					// Delete transient
					delete_transient('ejabat_register_'.$request['code']);
					// Success message
					$html = '<p class="ejabat"><span class="ejabat-info ejabat-success">'.sprintf(__('Your account %s has already been activated.', 'ejabberd-account-tools'),$transient['username'].'@'.$transient['host']).'</span></p>';
				}
			}
			// Code expired or not valid
			else {
				// Delete transient
				delete_transient('ejabat_register_'.$request['code']);
				// Message with error
				$message = '<p id="message" class="ejabat-info ejabat-blocked">'.__('The link to activate your account has expired or is invalid. Please complete the form and resubmit it.', 'ejabberd-account-tools').'</p>';
			}
		}
		// Get available host names
		$hosts = explode(' ', get_option('ejabat_registration_hosts', preg_replace('/^www\./','',$_SERVER['SERVER_NAME'])));
		// Foreach hosts
		$host_select = '';
		foreach($hosts as $host) {
			$host_select .= '<option value="' . $host . '" ' . selected($request['host'], $host, false) . '>@' . $host . '</option>';
		}
		$host_select = '<p id="host">
			<select name="host" '.wp_readonly(count($hosts), 1, false).'>
				'.$host_select.'
			</select>
		</p>';
		// Create form
		$html = isset($html) ? $html : '<form data-action="register" class="ejabat" method="post" onsubmit="return false" autocomplete="off">
			'.(isset($message) ? $message : '').'
			<p id="username" class="ejabat-validate ejabat-hint">
				<input type="text" name="username" placeholder="'.__('Username', 'ejabberd-account-tools').'">
				<span class="ejabat-tip"></span>
			</p>
			'.$host_select.'
			<p id="password" class="ejabat-strength ejabat-validate ejabat-hint">
				<input type="password" name="password" placeholder="'.__('Password', 'ejabberd-account-tools').'">
				<span class="ejabat-tip"></span>
			</p>
			<p id="email" class="ejabat-validate ejabat-hint">
				<input type="email" name="email" placeholder="'.__('Private email', 'ejabberd-account-tools').'">
				<span class="ejabat-tip"></span>
			</p>
			'.ejabat_captcha_field().'
			<p>
				<input type="submit" value="'.__('Create account', 'ejabberd-account-tools').'">
				<span class="ejabat-spinner" style="visibility: hidden;"></span>
			</p>
			<div id="response"></div>
		</form>';
	}
	return rest_ensure_response(array('data' => str_replace(array(PHP_EOL, "\t"), '', $html), 'nonce' => wp_create_nonce('wp_rest')));
}

// Route registration
function ejabat_route_register() {
	register_rest_route('ejabberd-account-tools/v1', '/register', array(
		'methods' => 'POST',
		'callback' => 'ejabat_register',
		'permission_callback' => '__return_true',
		'args' => array(
			'username' => array(
				'type' => 'string',
				'required' => true,
				'sanitize_callback' => function($param, $request, $key) {
					return stripslashes_deep(sanitize_text_field($param));
				}
			),
			'host' => array(
				'type' => 'string',
				'required' => true,
				'sanitize_callback' => function($param, $request, $key) {
					return stripslashes_deep(sanitize_text_field($param));
				}
			),
			'password' => array(
				'type' => 'string',
				'required' => true,
				'sanitize_callback' => function($param, $request, $key) {
					return stripslashes_deep(sanitize_text_field($param));
				}
			),
			'email' => array(
				'type' => 'string',
				'required' => true,
				'sanitize_callback' => function($param, $request, $key) {
					return stripslashes_deep(sanitize_text_field($param));
				}
			)
		)
	));
}
add_action('rest_api_init', 'ejabat_route_register');

// Registration form callback
function ejabat_register($request) {
	// Verify nonce
	if(!wp_verify_nonce($request->get_header('x-wp-nonce'), 'wp_rest')) {
		$status = 'blocked';
		$message = __('Verification error. Please try again.', 'ejabberd-account-tools');
	}
	else {
		// Verify registration timeout
		if(true == get_transient('ejabat_register_'.$_SERVER['REMOTE_ADDR'])) {
			$status = 'blocked';
			$message = __('You can\'t register another account so quickly. Please try again later.', 'ejabberd-account-tools');
		}
		else {
			// Verify captcha
			if(true !== ($captcha_verify = ejabat_captcha_verify($request))) {
				$status = 'blocked';
				if(is_array($captcha_verify)) $fields = $captcha_verify;
				$message = __('Captcha validation error. Please try again.', 'ejabberd-account-tools');
			}
			else {
				// Verify username
				if(!preg_match('/'.get_option('ejabat_allowed_username_regexp', '^(?=^[a-z0-9].*)(?=.*[a-z0-9]$)(?!.*([_.-])\1+.*)(?!^([0-9]+)$)[a-z0-9_.-]{3,30}$').'/i', $request['username'])) {
					$status = 'blocked';
					$fields = array('username');
					$message = __('The selected username doesn\'t meet the requirements. Please change it and try again.', 'ejabberd-account-tools');
				}
				else if(preg_match('/'.get_option('ejabat_blocked_username_regexp', '^(.*(404|abort|about|abuse|access|account|activat|address|adium|admin|adult|advertisin|affiliat|agile|ajax|allegro|analytics|android|anonymous|api|app|aqq|archiv|atom|auth|backup|billing|blog|board|bombus|bot|bug|business|cache|calendar|campaign|cancel|careers|cart|ceo|cgi|changelog|chat|check|chrome|client|cms|comercial|comment|compare|config|connect|contact|contest|contract|convers|cpp|creat|css|custome|dashboard|delete|demo|design|detail|develop|digsby|direct|disc|docs|document|domain|dot|drive|dropbox|ebay|ecommerce|edit|employment|enquiries|enterprise|error|event|facebook|faq|favorite|feed|file|firefox|flock|follow|form|forum|ftp|gadget|gajim|gist|github|google|group|guest|guide|help|homepage|host|htm|http|ijab|imap|index|info|instagram|instantbird|internal|intranet|invit|invoic|ipad|iphone|irc|irssi|issue|jabbear|jabber|jabbim|jabiru|jappix|java|jitsi|job|joomla|json|kadu|kopete|language|load|login|logout|logs|mail|manager|manual|market|media|member|message|messenger|microblog|microsoft|miranda|mobile|mozilla|mp3|msg|msn|mysql|name|network|news|nick|noreply|ns1|ns2|ns3|ns4|oauth|offers|office|olx|online|openid|operator|oracle|order|organizat|owner|page|pandion|panel|password|perl|php|pidgin|plugin|pop3|popular|porn|post|press|print|privacy|profil|promo|psi|pub|python|query|random|recruit|register|registrat|remove|replies|root|rss|ruby|sales|sample|save|script|search|secure|security|send|seo|service|session|setting|setup|shop|signin|signup|site|smtp|sql|ssh|ssl|staff|start|static|stats|status|store|subscrib|support|sysop|system|tablet|talk|task|team|tech|telnet|terms|test|theme|tigase|tkabber|tlen|tmp|todo|tool|translat|trillian|troll|tube|twitt|update|url|usage|user|vendas|video|visitor|voice|weather|web|widget|windows|work|wtw|www|xabber|xml|xmpp|yaml|yaxim|yml).*)$').'/i', $request['username'])) {
					$status = 'blocked';
					$fields = array('username');
					$message = __('The selected username contains forbidden words. Please change it and try again.', 'ejabberd-account-tools');
				}
				else {
					// Verify email
					if(!filter_var($request['email'], FILTER_VALIDATE_EMAIL) || !ejabat_validate_email_mxrecord($request['email'])) {
						$status = 'blocked';
						$fields = array('email');
						$message = __('The email address is invalid. Please change it and try again.', 'ejabberd-account-tools');
					}
					else if(!ejabat_validate_email_usercheck_com($request['email'])) {
						$status = 'blocked';
						$fields = array('email');
						$message = __('Disposable email addresses are not allowed. Please change the entered email address and try again.', 'ejabberd-account-tools');
					}
					else {
						// Verify host
						$host = $request['host'];
						$hosts = explode(' ', get_option('ejabat_registration_hosts', preg_replace('/^www\./','',$_SERVER['SERVER_NAME'])));
						if(in_array($host, $hosts)) {
							//Register account
							$response = ejabat_get_xmpp_data('register', array('user' => $request['username'], 'host' => $host, 'password' => $request['password']));
							// Server unavailable
							if(is_null($response)) {
								$status = 'error';
								$message = __('The server is temporarily unavailable. Please try again later.', 'ejabberd-account-tools');
							}
							// Successfully registered
							else if($response['code'] == 0) {
								// Get current timestamp
								$now = time();
								// Set private email
								ejabat_get_xmpp_data('private_set', array('user' => $request['username'], 'host' => $host, 'element' => '<private xmlns=\'email\'>'.$request['email'].'</private>'));
								// Two-step registration
								if(get_option('ejabat_two_step_registration', false)) {
									// Set code transient
									$code = bin2hex(openssl_random_pseudo_bytes(16));
									$data = array('timestamp' => $now, 'ip' => $_SERVER['REMOTE_ADDR'], 'username' => $request['username'], 'host' => $host, 'email' => $request['email']);
									set_transient('ejabat_register_'.$code, $data, get_option('ejabat_activation_timeout', 3600));
									// Prevent login
									ejabat_get_xmpp_data('ban_account', array('user' => $request['username'], 'host' => $host, 'reason' => 'Activation required'));
									// Email data
									$subject = sprintf(__('Confirm your new account on %s', 'ejabberd-account-tools'), $host);
									$body = sprintf(__('Hey %s!<br><br>You have registered the account %s with this email address. To complete your registration, please click the following link:<br><br>%s<br><br>If you haven\'t created an account, simply ignore this email.<br><br>Best regards,<br>%s', 'ejabberd-account-tools'), $request['username'], $request['username'].'@'.$host, '<a href="'.explode('?', $_SERVER['HTTP_REFERER'])[0].'?code='.$code.'">'.explode('?', $_SERVER['HTTP_REFERER'])[0].'?code='.$code.'</a>', get_bloginfo('name'));
									$headers[] = 'From: '.get_bloginfo('name').' <'.get_option('admin_email').'>';
									$headers[] = 'Content-Type: text/html; charset=UTF-8';
									// Try send email
									if(wp_mail($request['username'].' <'.$request['email'].'>', $subject, $body, $headers)) {
										$status = 'success';
										$message = __('An email has been sent to you to complete your registration. It contains an activation link that you must click.', 'ejabberd-account-tools');
									}
									// Problem with sending email
									else {
										$status = 'error';
										$message = __('Failed to complete registration. Please contact the administrator.', 'ejabberd-account-tools');
									}
								}
								// Normal registration
								else {
									// Form status
									$status = 'success';
									$message = sprintf(__('Your account %s has been successfully registered.', 'ejabberd-account-tools'), $request['username'].'@'.$host);
									// Send welcome message
									if(get_option('ejabat_welcome_msg', false)) {
										// Get subject and body
										$welcome_msg = apply_filters('ejabat_welcome_msg_args', array(
											'subject' => get_option('ejabat_welcome_msg_subject'),
											'body' => get_option('ejabat_welcome_msg_body')
										));
										$welcome_msg['subject'] = htmlspecialchars(wp_strip_all_tags(do_shortcode($welcome_msg['subject'])));
										$welcome_msg['subject'] = str_replace(array('\r\n', '\\r\\n', '\n', '\\n', '\r', '\\r'), ' ', $welcome_msg['subject']);
										$welcome_msg['body'] = htmlspecialchars(wp_strip_all_tags(do_shortcode($welcome_msg['body'])));
										$welcome_msg['body'] = str_replace(array('\r\n', '\\r\\n', '\n', '\\n', '\r', '\\r'), "\n", $welcome_msg['body']);
										// Send message
										if($welcome_msg['subject']) {
											ejabat_get_xmpp_data('send_message', array('type' => 'normal', 'from' => $host, 'to' => $request['username'].'@'.$host, 'subject' => $welcome_msg['subject'], 'body' => $welcome_msg['body']));
										} else {
											ejabat_get_xmpp_data('send_message', array('type' => 'chat', 'from' => $host, 'to' => $request['username'].'@'.$host, 'subject' => '', 'body' => $welcome_msg['body']));
										}
									}
								}
								// Registration watcher
								if(get_option('ejabat_watch_registration', true) && get_option('ejabat_watcher')) {
									$watchers = explode(' ', get_option('ejabat_watcher'));
									foreach($watchers as $watcher) {
										ejabat_get_xmpp_data('send_message', array('type' => 'chat', 'from' => $host, 'to' => $watcher, 'subject' => '', 'body' => sprintf('[%s] The account %s with the email address %s was registered from IP address %s', wp_date('Y-m-d H:i:s', $now), $request['username'].'@'.$host, $request['email'], $_SERVER['REMOTE_ADDR'])));
									}
								}
								// Set registration timeout
								if(get_option('ejabat_registration_timeout', 3600)) {
									$data = array('timestamp' => $now, 'username' => $request['username'], 'host' => $host, 'email' => $request['email']);
									set_transient('ejabat_register_'.$_SERVER['REMOTE_ADDR'], $data, get_option('ejabat_registration_timeout', 3600));
								}
							}
							// Already registered
							else if($response['code'] == 1 || 409) {
								$status = 'blocked';
								$fields = array('username');
								$message = __('The selected username is already registered. Please change it and try again.', 'ejabberd-account-tools');
							}
						}
						// Invalid host
						else {
							$status = 'error';
							$fields = array('host');
							$message = __('Invalid hostname. Please try again.', 'ejabberd-account-tools');
						}
					}
				}
			}
		}
	}
	// Return response
	if(get_option('ejabat_debug', false) == false) return rest_ensure_response(array('status' => isset($status) ? $status : 'error', 'message' => isset($message) ? $message : __('An unexpected error occurred. Please try again.', 'ejabberd-account-tools'), 'fields' => isset($fields) ? $fields : null));
	else return rest_ensure_response(array('status' => isset($status) ? $status : 'error', 'message' => isset($message) ? $message : __('An unexpected error occurred. Please try again.', 'ejabberd-account-tools'), 'fields' => isset($fields) ? $fields : null, 'debug_message' => isset($response['body']) ? $response['body'] : null, 'debug_code' => isset($response['code']) ? $response['code'] : null, 'debug_command' => isset($response['command']) ? $response['command'] : null, 'debug_arguments' => isset($response['arguments']) ? $response['arguments'] : null));
}
