<?php
/**
 * Web Presence configuration form
 */

if(!defined('ABSPATH')) {
	exit;
}

// Enqueue styles & scripts
function ejabat_enqueue_webpresence_scripts() {
	global $post;
	if(is_a($post, 'WP_Post') && has_shortcode($post->post_content, 'ejabat_webpresence')) {
		wp_enqueue_style('huebee', EJABAT_DIR_URL.'css/huebee.min.css', array(), '2.1.1', 'all');
		wp_enqueue_style('ejabat', EJABAT_DIR_URL.'css/style.min.css', array(), EJABAT_VERSION, 'all');
		wp_enqueue_script('huebee', EJABAT_DIR_URL.'js/huebee.pkgd.min.js', array(), '2.2.1', true);
		wp_enqueue_script('js-cookie', EJABAT_DIR_URL.'js/js.cookie.min.js', array(), '3.0.1', true);
		wp_enqueue_script('ejabat-form', EJABAT_DIR_URL.'js/js.ejabat.form.min.js', array(), EJABAT_VERSION, true);
		wp_localize_script('ejabat-form', 'ejabat', array(
			'nonce' => ejabat_create_nonce(),
			'rest_api' => esc_url_raw(rest_url().'ejabberd-account-tools/v1/'),
			'empty_field' => __('Please complete the required field.', 'ejabberd-account-tools'),
			'empty_fields' => __('Verification errors occurred. Please check all fields and submit the form again.', 'ejabberd-account-tools'),
			'error' => __('An unexpected error occurred. Please try again.', 'ejabberd-account-tools'),
			'form_error' => '<p class="ejabat"><span class="ejabat-info ejabat-error">'.__('An unexpected error occurred. Please try again.', 'ejabberd-account-tools').'</span></p>'
		));
	}
}
add_action('wp_enqueue_scripts', 'ejabat_enqueue_webpresence_scripts');

// Web Presence shortcode
function ejabat_webpresence_shortcode() {
	return '<p data-action="webpresence-form" class="ejabat">'.(get_option('ejabat_loader', true) && true ? '<span class="ejabat-loader" title="'.__('Loading', 'ejabberd-account-tools').'..."></span>' : '').'</p>';
}
add_shortcode('ejabat_webpresence', 'ejabat_webpresence_shortcode');

// Route Web Presence login and configuration form
function ejabat_route_webpresence_form() {
	register_rest_route('ejabberd-account-tools/v1', '/webpresence-form', array(
		'methods' => 'POST',
		'callback' => 'ejabat_webpresence_form',
		'permission_callback' => '__return_true',
		'args' => array(
			'session' => array(
				'type' => 'string',
				'required' => true,
				'default' => 'undefined',
				'sanitize_callback' => function($param, $request, $key) {
					return stripslashes_deep(sanitize_text_field($param));
				}
			)
		)
	));
}
add_action('rest_api_init', 'ejabat_route_webpresence_form');

// Web Presence login and configuration form callback
function ejabat_webpresence_form($request) {
	// Form is disabled
	if(get_option('ejabat_disable_webpresence', false) && !is_user_logged_in()) {
		$html = '<p class="ejabat"><span class="ejabat-info ejabat-error">'.__('Web Presence configuration form is temporarily disabled. Please try again later.', 'ejabberd-account-tools').'</span></p>';
	}
	else {
		// Default messaage
		$message = '<p class="ejabat-info ejabat-success">'.__('Please log in to continue Web Presence configuration.', 'ejabberd-account-tools').'</p>';
		// Session cookie is set
		if($request['session'] != 'undefined') {
			// Session valid
			if(true == ($transient = get_transient('ejabat_session_'.$request['session']))) {
				// Get Web Presence user settings
				$response = ejabat_get_xmpp_data('private_get', array('user' => $transient['user'], 'host' => $transient['host'], 'element' => 'webpresence', 'ns' => 'css'));
				$xml_css = @simplexml_load_string(json_decode($response['body']))->webpresence;
				$css = $xml_css->enabled ?: 'false';
				$css_available = $xml_css->available ?: '#4CAF50';
				$css_chate = $xml_css->chate ?: '#8BC34A';
				$css_away = $xml_css->away ?: '#795548';
				$css_xa = $xml_css->xa ?: '#607D8B';
				$css_dnd = $xml_css->dnd ?: '#F44336';
				$css_unavailable = $xml_css->unavailable ?: '#9E9E9E';
				$response = ejabat_get_xmpp_data('private_get', array('user' => $transient['user'], 'host' => $transient['host'], 'element' => 'webpresence', 'ns' => 'json'));
				$json = @simplexml_load_string(json_decode($response['body']))->webpresence->enabled ?: 'false';
				$response = ejabat_get_xmpp_data('private_get', array('user' => $transient['user'], 'host' => $transient['host'], 'element' => 'webpresence', 'ns' => 'image'));
				$image = @simplexml_load_string(json_decode($response['body']))->webpresence->enabled ?: 'false';
				// Create configuration form
				$html = '<form data-action="configure-webpresence" class="ejabat" method="post" onsubmit="return false" autocomplete="off">
					<p class="ejabat-logout">'.sprintf(__('Logged in as %s', 'ejabberd-account-tools'), $transient['user'].'@'.$transient['host']).' (<a href target="_self">'.__('log out', 'ejabberd-account-tools').'</a>)</p>
					<p>'.__('What features do you want to enable?', 'ejabberd-account-tools').'</p>
					<p>
						<label><input name="css" type="checkbox" value="true" '.checked('true', $css, false).' />CSS <span><a href="'.rest_url().get_option('ejabat_webpresence_url', 'webpresence').'/css/'.$transient['user'].'@'.$transient['host'].'" target="_blank">'.rest_url().get_option('ejabat_webpresence_url', 'webpresence').'/css/'.$transient['user'].'@'.$transient['host'].'</a></span></label>
					</p>
					<p>
						<input type="text" name="css_available" class="ejabat-color" maxlength="7" size="7" value="'.$css_available.'"/>'.__('Available', 'ejabberd-account-tools').'
					</p>
					<p>
						<input type="text" name="css_chate" class="ejabat-color" maxlength="7" size="7" value="'.$css_chate.'"/>'.__('Chat', 'ejabberd-account-tools').'
					</p>
					<p>
						<input type="text" name="css_away" class="ejabat-color" maxlength="7" size="7" value="'.$css_away.'"/>'.__('Away', 'ejabberd-account-tools').'
					</p>
					<p>
						<input type="text" name="css_xa" class="ejabat-color" maxlength="7" size="7" value="'.$css_xa.'"/>'.__('XA', 'ejabberd-account-tools').'
					</p>
					<p>
						<input type="text" name="css_dnd" class="ejabat-color" maxlength="7" size="7" value="'.$css_dnd.'"/>'.__('DND', 'ejabberd-account-tools').'
					</p>
					<p>
						<input type="text" name="css_unavailable" class="ejabat-color" maxlength="7" size="7" value="'.$css_unavailable.'"/>'.__('Unavailable', 'ejabberd-account-tools').'
					</p>
					<p>
						<label><input name="json" type="checkbox" value="true" '.checked('true', $json, false).' />JSON <span><a href="'.rest_url().get_option('ejabat_webpresence_url', 'webpresence').'/json/'.$transient['user'].'@'.$transient['host'].'" target="_blank">'.rest_url().get_option('ejabat_webpresence_url', 'webpresence').'/json/'.$transient['user'].'@'.$transient['host'].'</a></span></label>
					</p>
					<p>
						<label><input name="image" type="checkbox" value="true" '.checked('true', $image, false).' />'.__('Image file', 'ejabberd-account-tools').' <span><a href="'.rest_url().get_option('ejabat_webpresence_url', 'webpresence').'/image/'.$transient['user'].'@'.$transient['host'].'" target="_blank">'.rest_url().get_option('ejabat_webpresence_url', 'webpresence').'/image/'.$transient['user'].'@'.$transient['host'].'</a></span></label>
					</p>
					<p>
						<input type="hidden" name="session" value="'.$request['session'].'">
						<input type="submit" value="'.__('Save settings', 'ejabberd-account-tools').'">
						<span class="ejabat-spinner" style="visibility: hidden;"></span>
					</p>
					<div id="response"></div>
				</form>';
			}
			// Session expired or not valid
			else {
				// Delete transient
				delete_transient('ejabat_session_'.$request['session']);
				$request['session'] = 'expired';
				// Response with error
				$message = '<p class="ejabat-info ejabat-blocked">'.__('Your session has expired. Please log in again to continue Web Presence configuration.', 'ejabberd-account-tools').'</p>';
			}
		}
		// Create login form
		$html = isset($html) ? $html : '<form data-action="login" class="ejabat" method="post" onsubmit="return false" autocomplete="off">
			'.(isset($message) ? $message : '').'
			<p id="login" class="ejabat-validate">
				<input type="text" name="login" placeholder="'.__('Full username', 'ejabberd-account-tools').'">
				<span class="ejabat-tip"></span>
			</p>
			<p id="password" class="ejabat-validate">
				<input type="password" name="password" placeholder="'.__('Password', 'ejabberd-account-tools').'">
				<span class="ejabat-tip"></span>
			</p>
			'.ejabat_captcha_field().'
			<p>
				<input type="submit" value="'.__('Log in', 'ejabberd-account-tools').'">
				<span class="ejabat-spinner" style="visibility: hidden;"></span>
			</p>
			<div id="response"></div>
		</form>';
	}
	// Return response
	return rest_ensure_response(array('data' => str_replace(array(PHP_EOL, "\t"), '', $html), 'session' => $request['session'], 'nonce' => wp_create_nonce('wp_rest')));
}

// Route Configure Web Presence
function ejabat_route_configure_webpresence() {
	register_rest_route('ejabberd-account-tools/v1', '/configure-webpresence', array(
		'methods' => 'POST',
		'callback' => 'ejabat_configure_webpresence',
		'permission_callback' => '__return_true',
		'args' => array(
			'session' => array(
				'type' => 'string',
				'required' => true,
				'sanitize_callback' => function($param, $request, $key) {
					return stripslashes_deep(sanitize_text_field($param));
				}
			),
			'css' => array(
				'type' => 'boolean',
				'required' => true,
				'default' => 'false'
			),
			'css_available' => array(
				'type' => 'string',
				'required' => true,
				'default' => '#4CAF50',
				'sanitize_callback' => function($param, $request, $key) {
					return strtoupper(sanitize_text_field($param));
				},
				'validate_callback' => function($param, $request, $key) {
					return ctype_xdigit(ltrim($param, '#'));
				}
			),
			'css_chate' => array(
				'type' => 'string',
				'required' => true,
				'default' => '#8BC34A',
				'sanitize_callback' => function($param, $request, $key) {
					return strtoupper(sanitize_text_field($param));
				},
				'validate_callback' => function($param, $request, $key) {
					return ctype_xdigit(ltrim($param, '#'));
				}
			),
			'css_away' => array(
				'type' => 'string',
				'required' => true,
				'default' => '#795548',
				'sanitize_callback' => function($param, $request, $key) {
					return strtoupper(sanitize_text_field($param));
				},
				'validate_callback' => function($param, $request, $key) {
					return ctype_xdigit(ltrim($param, '#'));
				}
			),
			'css_xa' => array(
				'type' => 'string',
				'required' => true,
				'default' => '#607D8B',
				'sanitize_callback' => function($param, $request, $key) {
					return strtoupper(sanitize_text_field($param));
				},
				'validate_callback' => function($param, $request, $key) {
					return ctype_xdigit(ltrim($param, '#'));
				}
			),
			'css_dnd' => array(
				'type' => 'string',
				'required' => true,
				'default' => '#F44336',
				'sanitize_callback' => function($param, $request, $key) {
					return strtoupper(sanitize_text_field($param));
				},
				'validate_callback' => function($param, $request, $key) {
					return ctype_xdigit(ltrim($param, '#'));
				}
			),
			'css_unavailable' => array(
				'type' => 'string',
				'required' => true,
				'default' => '#9E9E9E',
				'sanitize_callback' => function($param, $request, $key) {
					return strtoupper(sanitize_text_field($param));
				},
				'validate_callback' => function($param, $request, $key) {
					return ctype_xdigit(ltrim($param, '#'));
				}
			),
			'json' => array(
				'type' => 'boolean',
				'required' => true,
				'default' => 'false'
			),
			'image' => array(
				'type' => 'boolean',
				'required' => true,
				'default' => 'false'
			)
		)
	));
}
add_action('rest_api_init', 'ejabat_route_configure_webpresence');

// Configure Web Presence callback
function ejabat_configure_webpresence($request) {
	// Verify nonce
	if(!wp_verify_nonce($request->get_header('x-wp-nonce'), 'wp_rest')) {
		$status = 'blocked';
		$message = __('Verification error. Please try again.', 'ejabberd-account-tools');
	}
	else {
		// Session valid
		if(true == ($transient = get_transient('ejabat_session_'.$request['session']))) {
			// Try save CSS settings
			$response = ejabat_get_xmpp_data('private_set', array('user' => $transient['user'], 'host' => $transient['host'], 'element' => '<webpresence xmlns=\'css\'><enabled>'.json_encode($request['css']).'</enabled><available>'.$request['css_available'].'</available><chate>'.$request['css_chate'].'</chate><away>'.$request['css_away'].'</away><xa>'.$request['css_xa'].'</xa><dnd>'.$request['css_dnd'].'</dnd><unavailable>'.$request['css_unavailable'].'</unavailable></webpresence>'));
			// Server unavailable
			if(is_null($response)) {
				$status = 'error';
				$message = __('The server is temporarily unavailable. Please try again later.', 'ejabberd-account-tools');
			}
			// CSS settings saved
			else if($response['code'] == 0) {
				// Try save JSON settings
				$respone = ejabat_get_xmpp_data('private_set', array('user' => $transient['user'], 'host' => $transient['host'], 'element' => '<webpresence xmlns=\'json\'><enabled>'.json_encode($request['json']).'</enabled></webpresence>'));
				// Server unavailable
				if(is_null($response)) {
					$status = 'error';
					$message = __('The server is temporarily unavailable. Please try again later.', 'ejabberd-account-tools');
				}
				// JSON settings saved
				else if($response['code'] == 0) {
					// Try save image settings
					$respone = ejabat_get_xmpp_data('private_set', array('user' => $transient['user'], 'host' => $transient['host'], 'element' => '<webpresence xmlns=\'image\'><enabled>'.json_encode($request['image']).'</enabled></webpresence>'));
					// Server unavailable
					if(is_null($response)) {
						$status = 'error';
						$message = __('The server is temporarily unavailable. Please try again later.', 'ejabberd-account-tools');
					}
					// Image settings saved
					else if($response['code'] == 0) {
						// Save settings to cache
						$css_transient = array(
							'enabled' => json_encode($request['css']),
							'available' => $request['css_available'],
							'chate' => $request['css_chate'],
							'away' => $request['css_away'],
							'xa' => $request['css_xa'],
							'dnd' => $request['css_dnd'],
							'unavailable' => $request['css_unavailable']
						);
						set_transient('ejabat_webpresence_css_'.$transient['user'].'@'.$transient['host'], $css_transient);
						set_transient('ejabat_webpresence_json_'.$transient['user'].'@'.$transient['host'], json_encode($request['json']));
						set_transient('ejabat_webpresence_image_'.$transient['user'].'@'.$transient['host'], json_encode($request['image']));
						// All settings saved
						$status = 'success';
						$message = __('Your settings have been successfully saved.', 'ejabberd-account-tools');
					}
				}
			}
		}
		// Session expired or not valid
		else {
			delete_transient('ejabat_session_'.$request['session']);
			$status = 'blocked';
			$message = __('Your session has expired. Please log in again to continue Web Presence configuration.', 'ejabberd-account-tools');
		}
	}
	// Return response
	if(get_option('ejabat_debug', false) == false) return rest_ensure_response(array('status' => isset($status) ? $status : 'error', 'message' => isset($message) ? $message : __('An unexpected error occurred. Please try again.', 'ejabberd-account-tools')));
	else return rest_ensure_response(array('status' => isset($status) ? $status : 'error', 'message' => isset($message) ? $message : __('An unexpected error occurred. Please try again.', 'ejabberd-account-tools'), 'debug_message' => isset($response['body']) ? $response['body'] : null, 'debug_code' => isset($response['code']) ? $response['code'] : null, 'debug_command' => isset($response['command']) ? $response['command'] : null, 'debug_arguments' => isset($response['arguments']) ? $response['arguments'] : null));
}

// Route presence in CSS
function ejabat_route_webpresence_css() {
	register_rest_route(get_option('ejabat_webpresence_url', 'webpresence'), '/css/(?P<jid>\S+)', array(
		'methods' => 'GET',
		'callback' => 'ejabat_webpresence_css',
		'permission_callback' => '__return_true',
		'args' => array(
			'jid' => array(
				'validate_callback' => 'is_email'
			)
		)
	));
}
add_action('rest_api_init', 'ejabat_route_webpresence_css');

// Return presence in CSS
function ejabat_webpresence_css($request) {
	// Get JID from request uri
	list($user, $host) = explode('@', $request['jid']);
	// Get webpresence settins from cache
	if(false === ($transient = get_transient('ejabat_webpresence_css_'.$user.'@'.$host))) {
		// Get webpresence settins from XMPP server
		$response = ejabat_get_xmpp_data('private_get', array('user' => $user, 'host' => $host, 'element' => 'webpresence', 'ns' => 'css'));
		$xml = (array)@simplexml_load_string(json_decode($response['body']))->webpresence;
		// Save settings to cache
		$transient = array(
			'enabled' => $xml['enabled'] ?: 'false',
			'available' => $xml['available'] ?: '#4CAF50',
			'chate' => $xml['chate'] ?: '#8BC34A',
			'away' => $xml['away'] ?: '#795548',
			'xa' => $xml['xa'] ?: '#607D8B',
			'dnd' => $xml['dnd'] ?: '#F44336',
			'unavailable' => $xml['unavailable'] ?: '#9E9E9E'
		);
		set_transient('ejabat_webpresence_css_'.$user.'@'.$host, $transient);
	}
	// Get presence
	if($transient['enabled'] == 'true') {
		$response = ejabat_get_xmpp_data('get_presence', array('user' => $user, 'host' => $host));
	}
	else {
		$rest_response = new WP_REST_Response(array('code' => 'rest_error', 'message' => 'Unauthorized', 'data' => array('status' => 401)));
		$rest_response->set_status(401);
		return $rest_response;
	}
	// Return response
	if(is_null($response)) {
		$rest_response = new WP_REST_Response(array('code' => 'rest_error', 'message' => 'Service Unavailable', 'data' => array('status' => 503)));
		$rest_response->set_status(503);
		return $rest_response;
	}
	else if($response['code'] == 0) {
		// Cache header
		header('Expires: '.gmdate('d M Y H:i:s', time() + 300).' GMT');
		header('Pragma: cache');
		header('Cache-Control: max-age=300');
		// Content header
		header('Content-Type: text/css');
		// Content
		$show = json_decode($response['body'])->show;
		if($show == 'available') $color = $transient['available'];
		else if($show == 'chat') $color = $transient['chat'];
		else if($show == 'away') $color = $transient['away'];
		else if($show == 'xa') $color = $transient['xa'];
		else if($show == 'dnd') $color = $transient['dnd'];
		else $color = $transient['unavailable'];
		echo '#webpresence{color:'.$color.';}';
	}
	else if($response['code'] == 1) {
		$rest_response = new WP_REST_Response(array('code' => 'rest_error', 'message' => 'Internal Server Error', 'data' => array('status' => 500)));
		$rest_response->set_status(500);
		return $rest_response;
	}
	else {
		$rest_response = new WP_REST_Response(array('code' => 'rest_error', 'message' => $response['body'], 'data' => array('status' => $response['code'])));
		$rest_response->set_status($response['code']);
		return $rest_response;
	}
}

// Route presence in JSON
function ejabat_route_webpresence_json() {
	register_rest_route(get_option('ejabat_webpresence_url', 'webpresence'), '/json/(?P<jid>\S+)', array(
		'methods' => 'GET',
		'callback' => 'ejabat_webpresence_json',
		'permission_callback' => '__return_true',
		'args' => array(
			'jid' => array(
				'validate_callback' => 'is_email'
			)
		)
	));
}
add_action('rest_api_init', 'ejabat_route_webpresence_json');

// Return presence in JSON
function ejabat_webpresence_json($request) {
	// Get JID from request uri
	list($user, $host) = explode('@', $request['jid']);
	// Get webpresence settins from cache
	if(false === ($transient = get_transient('ejabat_webpresence_json_'.$user.'@'.$host))) {
		// Get webpresence settins from XMPP server
		$response = ejabat_get_xmpp_data('private_get', array('user' => $user, 'host' => $host, 'element' => 'webpresence', 'ns' => 'json'));
		$xml = (array)@simplexml_load_string(json_decode($response['body']))->webpresence;
		// Save settings to cache
		$transient = $xml['enabled'] ?: 'false';
		set_transient('ejabat_webpresence_json_'.$user.'@'.$host, $transient);
	}
	// Get presence
	if($transient == 'true') {
		$response = ejabat_get_xmpp_data('get_presence', array('user' => $user, 'host' => $host));
	}
	else {
		$rest_response = new WP_REST_Response(array('code' => 'rest_error', 'message' => 'Unauthorized', 'data' => array('status' => 401)));
		$rest_response->set_status(401);
		return $rest_response;
	}
	// Return response
	if(is_null($response)) {
		$rest_response = new WP_REST_Response(array('code' => 'rest_error', 'message' => 'Service Unavailable', 'data' => array('status' => 503)));
		$rest_response->set_status(503);
		return $rest_response;
	}
	else if($response['code'] == 0) {
		// Cache header
		header('Expires: '.gmdate('d M Y H:i:s', time() + 60).' GMT');
		header('Pragma: cache');
		header('Cache-Control: max-age=60');
		// Return presence
		return rest_ensure_response(array('jid' => json_decode($response['body'])->jid, 'state' => json_decode($response['body'])->show, 'status' => json_decode($response['body'])->status));
	}
	else if($response['code'] == 1) {
		$rest_response = new WP_REST_Response(array('code' => 'rest_error', 'message' => 'Internal Server Error', 'data' => array('status' => 500)));
		$rest_response->set_status(500);
		return $rest_response;
	}
	else {
		$rest_response = new WP_REST_Response(array('code' => 'rest_error', 'message' => $response['body'], 'data' => array('status' => $response['code'])));
		$rest_response->set_status($response['code']);
		return $rest_response;
	}
}

// Route presence in image file
function ejabat_route_webpresence_image() {
	register_rest_route(get_option('ejabat_webpresence_url', 'webpresence'), '/image/(?P<jid>\S+)', array(
		'methods' => 'GET',
		'callback' => 'ejabat_webpresence_image',
		'permission_callback' => '__return_true',
		'args' => array(
			'jid' => array(
				'validate_callback' => 'is_email'
			)
		)
	));
}
add_action('rest_api_init', 'ejabat_route_webpresence_image');

// Return presence in image file
function ejabat_webpresence_image($request) {
	// Get JID from request uri
	list($user, $host) = explode('@', $request['jid']);
	// Get webpresence settins from cache
	if(false === ($transient = get_transient('ejabat_webpresence_image_'.$user.'@'.$host))) {
		// Get webpresence settins from XMPP server
		$response = ejabat_get_xmpp_data('private_get', array('user' => $user, 'host' => $host, 'element' => 'webpresence', 'ns' => 'image'));
		$xml = (array)@simplexml_load_string(json_decode($response['body']))->webpresence;
		// Save settings to cache
		$transient = $xml['enabled'] ?: 'false';
		set_transient('ejabat_webpresence_image_'.$user.'@'.$host, $transient);
	}
	// Get presence
	if($transient == 'true') {
		$response = ejabat_get_xmpp_data('get_presence', array('user' => $user, 'host' => $host));
	}
	else {
		$rest_response = new WP_REST_Response(array('code' => 'rest_error', 'message' => 'Unauthorized', 'data' => array('status' => 401)));
		$rest_response->set_status(401);
		return $rest_response;
	}
	// Return response
	if(is_null($response)) {
		$rest_response = new WP_REST_Response(array('code' => 'rest_error', 'message' => 'Service Unavailable', 'data' => array('status' => 503)));
		$rest_response->set_status(503);
		return $rest_response;
	}
	else if($response['code'] == 0) {
		// Cache header
		header('Expires: '.gmdate('d M Y H:i:s', time() + 300).' GMT');
		header('Pragma: cache');
		header('Cache-Control: max-age=300');
		// Content
		$show = json_decode($response['body'])->show;
		if(true == ($file = glob(get_stylesheet_directory().'/webpresence/'.$show.'.*'))) {
			header('Content-Type: '.getimagesize($file[0])['mime']);
			readfile($file[0]);
		} elseif(true == ($file = glob(get_template_directory().'/webpresence/'.$show.'.*'))) {
			header('Content-Type: '.getimagesize($file[0])['mime']);
			readfile($file[0]);
		} else {
			header('Content-Type: image/png');
			readfile(EJABAT_DIR_URL.'img/'.$show.'.png');
		}
	}
	else if($response['code'] == 1) {
		$rest_response = new WP_REST_Response(array('code' => 'rest_error', 'message' => 'Internal Server Error', 'data' => array('status' => 500)));
		$rest_response->set_status(500);
		return $rest_response;
	}
	else {
		$rest_response = new WP_REST_Response(array('code' => 'rest_error', 'message' => $response['body'], 'data' => array('status' => $response['code'])));
		$rest_response->set_status($response['code']);
		return $rest_response;
	}
}